<?php
/**
 * Mesaj Polling API - WebSocket alternatifi
 * Bu dosya gerçek zamanlı mesaj kontrolü için kullanılır
 */

require_once '../config/database.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');
header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');

// Kullanıcı giriş kontrolü
session_start();
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit();
}

try {
    $user_id = $_SESSION['user_id'];
    $last_check = $_GET['last_check'] ?? date('Y-m-d H:i:s', strtotime('-1 minute'));
    
    // Son kontrol zamanından sonra gelen mesajları getir
    $stmt = $pdo->prepare("
        SELECT m.*, c.id as conversation_id, cu.name as customer_name, cu.phone as customer_phone
        FROM messages m
        JOIN conversations c ON m.conversation_id = c.id
        JOIN customers cu ON c.customer_id = cu.id
        WHERE m.created_at > ? 
        AND m.sender_type = 'customer'
        AND m.is_read = 0
        AND (c.assigned_to = ? OR ? IN (SELECT id FROM users WHERE role IN ('admin', 'supervisor')))
        ORDER BY m.created_at ASC
    ");
    
    $user = getUserById($user_id);
    $isAdmin = in_array($user['role'], ['admin', 'supervisor']);
    $stmt->execute([$last_check, $user_id, $user_id]);
    $newMessages = $stmt->fetchAll();
    
    // Yeni sohbet sayısını getir
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as new_conversations
        FROM conversations c
        WHERE c.created_at > ? 
        AND c.assigned_to = ?
    ");
    $stmt->execute([$last_check, $user_id]);
    $newConversations = $stmt->fetch()['new_conversations'];
    
    // Aktif sohbet sayısını getir
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as active_conversations
        FROM conversations c
        WHERE c.assigned_to = ? AND c.status = 'active'
    ");
    $stmt->execute([$user_id]);
    $activeConversations = $stmt->fetch()['active_conversations'];
    
    // Bildirimleri formatla
    $notifications = [];
    foreach ($newMessages as $message) {
        $notifications[] = [
            'type' => 'new_message',
            'conversation_id' => $message['conversation_id'],
            'customer_name' => $message['customer_name'],
            'customer_phone' => $message['customer_phone'],
            'message' => $message['content'],
            'time' => timeAgo($message['created_at']),
            'timestamp' => $message['created_at']
        ];
    }
    
    echo json_encode([
        'success' => true,
        'timestamp' => date('Y-m-d H:i:s'),
        'notifications' => $notifications,
        'stats' => [
            'new_messages' => count($newMessages),
            'new_conversations' => $newConversations,
            'active_conversations' => $activeConversations
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
