<?php
session_start();
require_once '../config/database.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

// Log function
function logFBSend($message) {
    $logFile = __DIR__ . '/../logs/facebook_send_' . date('Y-m-d') . '.log';
    $logMessage = '[' . date('Y-m-d H:i:s') . '] ' . $message . PHP_EOL;
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

logFBSend("=== Send Message Request ===");

// Kullanıcı girişi kontrolü
if (!isset($_SESSION['user_id'])) {
    logFBSend("ERROR: Unauthorized");
    echo json_encode(['success' => false, 'error' => 'Yetkisiz erişim.']);
    exit();
}

$user_id = $_SESSION['user_id'];
$user = getUserById($user_id);

// POST verilerini al
$input = json_decode(file_get_contents('php://input'), true);
$conversation_id = $input['conversation_id'] ?? null;
$message = $input['message'] ?? '';

logFBSend("User: $user_id, Conversation: $conversation_id");

if (!$conversation_id || !$message) {
    logFBSend("ERROR: Missing parameters");
    echo json_encode(['success' => false, 'error' => 'Eksik parametreler.']);
    exit();
}

try {
    // Sohbet bilgilerini al
    $conversation = getConversationById($conversation_id);
    if (!$conversation) {
        logFBSend("ERROR: Conversation not found");
        echo json_encode(['success' => false, 'error' => 'Sohbet bulunamadı.']);
        exit();
    }
    
    // Müşteri bilgilerini al
    $customer = getCustomerById($conversation['customer_id']);
    if (!$customer) {
        logFBSend("ERROR: Customer not found");
        echo json_encode(['success' => false, 'error' => 'Müşteri bulunamadı.']);
        exit();
    }
    
    logFBSend("Customer: " . $customer['name'] . ", FB ID: " . ($customer['facebook_id'] ?? 'NULL'));
    
    // Facebook ID kontrolü
    if (!$customer['facebook_id']) {
        logFBSend("ERROR: No Facebook ID");
        echo json_encode(['success' => false, 'error' => 'Müşterinin Facebook ID\'si bulunamadı.']);
        exit();
    }
    
    // Check if facebook_config is loaded
    global $facebook_config;
    if (!isset($facebook_config)) {
        logFBSend("ERROR: Facebook config not loaded");
        echo json_encode(['success' => false, 'error' => 'Facebook yapılandırması yüklenmedi.']);
        exit();
    }
    
    logFBSend("Config loaded, Page Token: " . substr($facebook_config['page_access_token'], 0, 20) . "...");
    
    // Facebook mesajı gönder
    $result = sendFacebookMessage($customer['facebook_id'], $message);
    
    logFBSend("Send result - Success: " . ($result['success'] ? 'YES' : 'NO') . ", HTTP: " . $result['http_code']);
    logFBSend("Response: " . json_encode($result['response']));
    
    if ($result['success']) {
        // Mesajı veritabanına kaydet
        $message_id = saveMessage([
            'conversation_id' => $conversation_id,
            'sender_id' => $user_id,
            'content' => $message,
            'sender_type' => 'agent',
            'facebook_message_id' => $result['response']['message_id'] ?? ''
        ]);
        
        logFBSend("Message saved to DB: $message_id");
        
        echo json_encode([
            'success' => true,
            'message_id' => $message_id,
            'response' => $result['response']
        ]);
    } else {
        $errorMsg = $result['response']['error']['message'] ?? 'Bilinmeyen hata';
        logFBSend("ERROR: " . $errorMsg);
        
        echo json_encode([
            'success' => false,
            'error' => 'Facebook mesajı gönderilemedi: ' . $errorMsg,
            'debug' => $result['response']
        ]);
    }
    
} catch (Exception $e) {
    logFBSend("EXCEPTION: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'error' => 'Bir hata oluştu: ' . $e->getMessage()
    ]);
}
?>
