<?php
/**
 * Send Instagram Message API
 * Sends messages via Instagram Graph API
 */

session_start();
require_once '../config/database.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

// Log function
function logIGSend($message) {
    $logFile = __DIR__ . '/../logs/instagram_send_' . date('Y-m-d') . '.log';
    $logMessage = '[' . date('Y-m-d H:i:s') . '] ' . $message . PHP_EOL;
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

logIGSend("=== Send Instagram Message Request ===");

// Auth check
if (!isset($_SESSION['user_id'])) {
    logIGSend("ERROR: Unauthorized");
    echo json_encode(['success' => false, 'error' => 'Yetkisiz erişim.']);
    exit();
}

$user_id = $_SESSION['user_id'];
$user = getUserById($user_id);

// Get POST data
$input = json_decode(file_get_contents('php://input'), true);
$conversation_id = $input['conversation_id'] ?? null;
$message = $input['message'] ?? '';

logIGSend("User: $user_id, Conversation: $conversation_id");

if (!$conversation_id || !$message) {
    logIGSend("ERROR: Missing parameters");
    echo json_encode(['success' => false, 'error' => 'Eksik parametreler.']);
    exit();
}

try {
    // Get conversation
    $conversation = getConversationById($conversation_id);
    if (!$conversation) {
        logIGSend("ERROR: Conversation not found");
        echo json_encode(['success' => false, 'error' => 'Sohbet bulunamadı.']);
        exit();
    }
    
    // Get customer
    $customer = getCustomerById($conversation['customer_id']);
    if (!$customer) {
        logIGSend("ERROR: Customer not found");
        echo json_encode(['success' => false, 'error' => 'Müşteri bulunamadı.']);
        exit();
    }
    
    logIGSend("Customer: " . $customer['name'] . ", IG ID: " . ($customer['instagram_id'] ?? 'NULL'));
    
    // Check Instagram ID
    if (!$customer['instagram_id']) {
        logIGSend("ERROR: No Instagram ID");
        echo json_encode(['success' => false, 'error' => 'Müşterinin Instagram ID\'si bulunamadı.']);
        exit();
    }
    
    // Check if facebook_config is loaded (Instagram uses same config)
    global $facebook_config;
    if (!isset($facebook_config)) {
        logIGSend("ERROR: Facebook config not loaded");
        echo json_encode(['success' => false, 'error' => 'Instagram yapılandırması yüklenmedi.']);
        exit();
    }
    
    logIGSend("Config loaded, Page Token: " . substr($facebook_config['page_access_token'], 0, 20) . "...");
    
    // Send Instagram message (uses Facebook Graph API)
    $url = "https://graph.facebook.com/{$facebook_config['api_version']}/me/messages";
    
    $data = [
        'recipient' => ['id' => $customer['instagram_id']],
        'message' => ['text' => $message]
    ];
    
    $headers = [
        'Authorization: Bearer ' . $facebook_config['page_access_token'],
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    $result = [
        'success' => $httpCode === 200,
        'response' => json_decode($response, true),
        'http_code' => $httpCode
    ];
    
    logIGSend("Send result - Success: " . ($result['success'] ? 'YES' : 'NO') . ", HTTP: " . $result['http_code']);
    logIGSend("Response: " . json_encode($result['response']));
    
    if ($result['success']) {
        // Save message to database
        $message_id = saveMessage([
            'conversation_id' => $conversation_id,
            'sender_id' => $user_id,
            'content' => $message,
            'sender_type' => 'agent',
            'instagram_message_id' => $result['response']['message_id'] ?? ''
        ]);
        
        logIGSend("Message saved to DB: $message_id");
        
        echo json_encode([
            'success' => true,
            'message_id' => $message_id,
            'response' => $result['response']
        ]);
    } else {
        $errorMsg = $result['response']['error']['message'] ?? 'Bilinmeyen hata';
        logIGSend("ERROR: " . $errorMsg);
        
        echo json_encode([
            'success' => false,
            'error' => 'Instagram mesajı gönderilemedi: ' . $errorMsg,
            'debug' => $result['response']
        ]);
    }
    
} catch (Exception $e) {
    logIGSend("EXCEPTION: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'error' => 'Bir hata oluştu: ' . $e->getMessage()
    ]);
}
?>
