<?php
/**
 * WhatsApp Mesaj Gönderme API
 */

require_once '../config/database.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

// Sadece POST isteklerini kabul et
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

// Kullanıcı giriş kontrolü
session_start();
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit();
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $conversationId = $input['conversation_id'] ?? null;
    $message = $input['message'] ?? '';
    $type = $input['type'] ?? 'text';
    
    if (!$conversationId || !$message) {
        throw new Exception('Gerekli parametreler eksik');
    }
    
    // Sohbeti kontrol et
    $conversation = getConversationById($conversationId);
    if (!$conversation) {
        throw new Exception('Sohbet bulunamadı');
    }
    
    // Kullanıcının bu sohbeti yönetme yetkisi var mı?
    if ($conversation['assigned_to'] != $_SESSION['user_id']) {
        // Admin veya supervisor ise izin ver
        $user = getUserById($_SESSION['user_id']);
        if (!in_array($user['role'], ['admin', 'supervisor'])) {
            throw new Exception('Bu sohbeti yönetme yetkiniz yok');
        }
    }
    
    // WhatsApp'a mesaj gönder
    writeLog("WhatsApp mesaj gönderiliyor - Telefon: " . $conversation['customer_phone'] . " - Mesaj: " . $message, 'INFO');
    $result = sendWhatsAppMessage($conversation['customer_phone'], $message, $type);
    writeLog("WhatsApp API sonucu: " . json_encode($result), 'INFO');
    
    if ($result['success']) {
        // Mesajı veritabanına kaydet - yeni format
        $messageId = $result['response']['messages'][0]['id'] ?? null;
        saveMessage([
            'conversation_id' => $conversationId,
            'sender_id' => $_SESSION['user_id'],
            'content' => $message,
            'sender_type' => 'agent',
            'whatsapp_message_id' => $messageId
        ]);
        
        // Sohbeti güncelle
        $stmt = $pdo->prepare("UPDATE conversations SET updated_at = NOW() WHERE id = ?");
        $stmt->execute([$conversationId]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Mesaj başarıyla gönderildi',
            'whatsapp_message_id' => $messageId
        ]);
        
        writeLog("Mesaj gönderildi - Sohbet: " . $conversationId . " - Mesaj: " . $message);
    } else {
        $errorMsg = 'Bilinmeyen hata';
        if (isset($result['response']['error']['message'])) {
            $errorMsg = $result['response']['error']['message'];
        } elseif (isset($result['error'])) {
            $errorMsg = $result['error'];
        }
        
        writeLog("WhatsApp API hatası: " . json_encode($result), 'ERROR');
        throw new Exception('WhatsApp API hatası: ' . $errorMsg);
    }
    
} catch (Exception $e) {
    $errorMessage = $e->getMessage();
    $errorTrace = $e->getTraceAsString();
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $errorMessage,
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $errorTrace
        ]
    ]);
    
    writeLog("Mesaj gönderme hatası: " . $errorMessage . "\n" . $errorTrace, 'ERROR');
}
?>
