/**
 * MultiCRM - WhatsApp CRM Sistemi JavaScript
 */

class MultiCRM {
    constructor() {
        this.currentConversationId = null;
        this.messagePollingInterval = null;
        this.eventSource = null;
        this.lastMessageTime = null;
        this.updateDebounce = null;
        this.pendingUpdates = {
            messages: false,
            conversations: false
        };
        this.init();
    }

    init() {
        this.bindEvents();
        // Bildirim izni isteme - kullanıcıyı rahatsız etmesin
        this.startPolling(); // Polling kullan - daha güvenilir

        // Butonları başlangıçta gizle
        document.getElementById('customer-info-btn').style.display = 'none';
        document.getElementById('delete-conversation-btn').style.display = 'none';
    }

    requestNotificationPermission() {
        // Bildirim izni devre dışı - kullanıcı deneyimi için
        // İsteyen kullanıcılar tarayıcı ayarlarından açabilir
    }

    bindEvents() {
        // Yeni sohbet modalı
        document.getElementById('create-conversation-btn')?.addEventListener('click', () => {
            this.createNewConversation();
        });

        // Mesaj gönderme
        document.getElementById('send-message-btn')?.addEventListener('click', () => {
            this.sendMessage();
        });

        document.getElementById('message-input')?.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.sendMessage();
            }
        });

        // Sohbet seçimi
        document.addEventListener('click', (e) => {
            if (e.target.closest('.conversation-item')) {
                const conversationId = e.target.closest('.conversation-item').dataset.conversationId;
                this.selectConversation(conversationId);
            }
        });
    }

    async createNewConversation() {
        const phone = document.getElementById('customer-phone').value;
        const name = document.getElementById('customer-name').value;

        if (!phone) {
            this.showNotification('Telefon numarası gerekli', 'error');
            return;
        }

        try {
            const response = await fetch('api/create-conversation.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    phone: phone,
                    name: name
                })
            });

            const result = await response.json();

            if (result.success) {
                this.showNotification('Sohbet oluşturuldu', 'success');
                bootstrap.Modal.getInstance(document.getElementById('newConversationModal')).hide();
                document.getElementById('new-conversation-form').reset();
                this.loadConversations();
                this.selectConversation(result.conversation_id);
            } else {
                this.showNotification(result.error || 'Sohbet oluşturulamadı', 'error');
            }
        } catch (error) {
            console.error('Sohbet oluşturma hatası:', error);
            this.showNotification('Bir hata oluştu', 'error');
        }
    }

    async selectConversation(conversationId) {
        if (this.currentConversationId === conversationId) return;

        this.currentConversationId = conversationId;

        // Aktif sohbeti işaretle
        document.querySelectorAll('.conversation-item').forEach(item => {
            item.classList.remove('active');
        });
        document.querySelector(`[data-conversation-id="${conversationId}"]`)?.classList.add('active');

        // Önceki mesajları temizle
        const chatMessages = document.getElementById('chat-messages');
        chatMessages.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i> Yükleniyor...</div>';

        // Mesajları yükle
        await this.loadMessages(conversationId);
    }

    async loadMessages(conversationId) {
        console.log('🔵 loadMessages çağrıldı, conversationId:', conversationId);
        try {
            const response = await fetch(`api/get-messages.php?conversation_id=${conversationId}`);
            const result = await response.json();
            console.log('🔵 API yanıtı:', result);

            if (result.success) {
                console.log('✅ Mesajlar yüklendi, sayı:', result.messages.length);
                // Update conversation info
                if (result.conversation) {
                    this.updateChatHeader(result.conversation);

                    // AI Toggle Update
                    const aiToggle = document.getElementById('ai-toggle');
                    if (aiToggle) {
                        aiToggle.disabled = false;
                        aiToggle.checked = result.conversation.is_ai_active;

                        // Remove old listener to prevent duplicates (cloning is a simple way)
                        const newToggle = aiToggle.cloneNode(true);
                        aiToggle.parentNode.replaceChild(newToggle, aiToggle);

                        newToggle.addEventListener('change', async (e) => {
                            const isActive = e.target.checked;
                            try {
                                const response = await fetch('api/toggle-ai.php', {
                                    method: 'POST',
                                    headers: { 'Content-Type': 'application/json' },
                                    body: JSON.stringify({
                                        conversation_id: this.currentConversationId,
                                        is_active: isActive
                                    })
                                });
                                const toggleResult = await response.json(); // Renamed to avoid conflict with outer 'result'
                                if (toggleResult.success) {
                                    this.showNotification(isActive ? 'AI Asistanı Aktif' : 'AI Asistanı Pasif', 'success');
                                } else {
                                    e.target.checked = !isActive; // Revert
                                    this.showNotification('AI durumu değiştirilemedi', 'error');
                                }
                            } catch (error) {
                                e.target.checked = !isActive; // Revert
                                this.showNotification('Hata oluştu', 'error');
                            }
                        });
                    }

                    // Aktif chatbot'u otomatik ata ve AI'yı kontrol et
                    fetch('api/chatbots/get-active.php')
                        .then(res => res.json())
                        .then(data => {
                            if (data.success && data.chatbot) {
                                // Aktif chatbot var
                                if (!result.conversation.chatbot_id) {
                                    // Sohbete chatbot ata
                                    fetch('api/chatbots/set-conversation-bot.php', {
                                        method: 'POST',
                                        headers: { 'Content-Type': 'application/json' },
                                        body: JSON.stringify({
                                            conversation_id: this.currentConversationId,
                                            chatbot_id: data.chatbot.id
                                        })
                                    });
                                }

                                // Zamanlama kontrolü yap ve AI'yı aktif et
                                fetch('api/chatbots/check-schedule.php', {
                                    method: 'POST',
                                    headers: { 'Content-Type': 'application/json' },
                                    body: JSON.stringify({
                                        conversation_id: this.currentConversationId
                                    })
                                })
                                    .then(res => res.json())
                                    .then(scheduleData => {
                                        const aiToggle = document.getElementById('ai-toggle');
                                        if (scheduleData.should_be_active) {
                                            // Zamanlama içinde, AI'yı aktif et
                                            if (aiToggle && !aiToggle.checked) {
                                                aiToggle.checked = true;
                                            }
                                        } else {
                                            // Zamanlama dışında, AI'yı pasif et
                                            if (aiToggle && aiToggle.checked) {
                                                aiToggle.checked = false;
                                            }
                                        }
                                    });
                            } else {
                                // Aktif chatbot yok, AI'yı pasif yap
                                const aiToggle = document.getElementById('ai-toggle');
                                if (aiToggle && aiToggle.checked) {
                                    aiToggle.checked = false;
                                    fetch('api/toggle-ai.php', {
                                        method: 'POST',
                                        headers: { 'Content-Type': 'application/json' },
                                        body: JSON.stringify({
                                            conversation_id: this.currentConversationId,
                                            is_active: false
                                        })
                                    });
                                }
                            }
                        });

                    // Customer Status Selector'ı göster ve mevcut durumu yükle
                    const statusWrapper = document.getElementById('customer-status-wrapper');
                    const statusSelector = document.getElementById('customer-status-selector');
                    if (statusWrapper && statusSelector) {
                        statusWrapper.style.display = 'block';
                        statusSelector.value = result.conversation.customer_status || '';

                        // Status değiştiğinde kaydet
                        statusSelector.onchange = async (e) => {
                            const newStatus = e.target.value;
                            try {
                                const response = await fetch('api/update-customer-status.php', {
                                    method: 'POST',
                                    headers: { 'Content-Type': 'application/json' },
                                    body: JSON.stringify({
                                        conversation_id: this.currentConversationId,
                                        customer_status: newStatus
                                    })
                                });
                                const updateStatusResult = await response.json(); // Renamed to avoid conflict
                                if (updateStatusResult.success) {
                                    this.showNotification('Müşteri durumu güncellendi', 'success');
                                } else {
                                    this.showNotification(updateStatusResult.error || 'Müşteri durumu güncellenemedi', 'error');
                                }
                            } catch (error) {
                                console.error('Durum güncellenemedi:', error);
                                this.showNotification('Hata oluştu: Müşteri durumu güncellenemedi', 'error');
                            }
                        };
                    }
                }
                this.displayMessages(result.messages);
                this.showChatInput();
            } else {
                console.error('❌ API hatası:', result.error);
                this.showNotification(result.error || 'Mesajlar yüklenemedi', 'error');
            }
        } catch (error) {
            console.error('❌ Mesaj yükleme hatası:', error);
            this.showNotification('Mesajlar yüklenemedi', 'error');
        }
    }

    displayMessages(messages) {
        const chatMessages = document.getElementById('chat-messages');

        if (messages.length === 0) {
            chatMessages.innerHTML = '<div class="no-chat-selected"><i class="fas fa-comments"></i><p>Henüz mesaj yok</p></div>';
            return;
        }

        // İlk yükleme veya loading durumu: tüm mesajları göster
        const isInitialLoad = chatMessages.children.length === 0 ||
            chatMessages.innerHTML.includes('no-chat-selected') ||
            chatMessages.innerHTML.includes('loading');

        if (isInitialLoad) {
            chatMessages.innerHTML = '';
            messages.forEach(message => {
                this.addMessageElement(chatMessages, message);
            });
        } else {
            // Incremental update: sadece yeni mesajları ekle
            const existingIds = new Set(
                [...chatMessages.querySelectorAll('.message[data-message-id]')]
                    .map(el => String(el.dataset.messageId))
            );

            // Optimistic update mesajlarını temizle (messageId olmayan)
            const optimisticMessages = chatMessages.querySelectorAll('.message:not([data-message-id])');
            optimisticMessages.forEach(msg => msg.remove());

            // Yeni mesajları ekle
            messages.forEach(message => {
                if (!existingIds.has(String(message.id))) {
                    this.addMessageElement(chatMessages, message);
                }
            });
        }

        // En alta kaydır
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    addMessageElement(container, message) {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${message.sender_type === 'agent' ? 'sent' : 'received'}`;
        messageDiv.dataset.messageId = message.id;

        messageDiv.innerHTML = `
            <div class="message-bubble">
                ${this.escapeHtml(message.content)}
                <div class="message-time">${message.time_ago}</div>
            </div>
        `;

        container.appendChild(messageDiv);
    }

    // updateChatHeader yalnızca dosyanın sonunda tanımlanıyor (bkz satır 484)

    showChatInput() {
        document.getElementById('chat-input-area').style.display = 'block';
        document.getElementById('message-input').focus();
    }

    async sendMessage() {
        const messageInput = document.getElementById('message-input');
        const message = messageInput.value.trim();

        if (!message || !this.currentConversationId) return;

        console.log('📤 Mesaj gönderiliyor:', message);

        // Mesaj inputunu temizle ve disable et
        messageInput.value = '';
        messageInput.disabled = true;

        try {
            // Platform belirleme - mevcut sohbet bilgisini al
            const currentConversation = document.querySelector(`[data-conversation-id="${this.currentConversationId}"]`);
            const avatarDiv = currentConversation?.querySelector('.conversation-avatar');

            let platform = 'whatsapp'; // default
            let apiEndpoint = 'api/send-message.php';

            if (avatarDiv) {
                if (avatarDiv.classList.contains('facebook')) {
                    platform = 'facebook';
                    apiEndpoint = 'api/send-facebook-message.php';
                } else if (avatarDiv.classList.contains('instagram')) {
                    platform = 'instagram';
                    apiEndpoint = 'api/send-instagram-message.php';
                }
            }

            console.log('Platform:', platform, 'API:', apiEndpoint);

            const response = await fetch(apiEndpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    conversation_id: this.currentConversationId,
                    message: message,
                    type: 'text'
                })
            });

            const result = await response.json();
            console.log('📤 API yanıtı:', result);

            if (!result.success) {
                console.error('❌ HATA DETAYI:', result);
                console.error('❌ Hata mesajı:', result.error);
                if (result.debug) {
                    console.error('❌ Debug bilgisi:', result.debug);
                }
                this.showNotification(result.error || 'Mesaj gönderilemedi', 'error');
            } else {
                console.log('✅ Mesaj başarıyla gönderildi');
                // Başarılı - mesajları hemen yeniden yükle
                await this.loadMessages(this.currentConversationId);
                // Sohbet listesini güncelle (son mesaj önizlemesi için)
                this.loadConversations();
            }
        } catch (error) {
            console.error('❌ Mesaj gönderme hatası:', error);
            this.showNotification('Mesaj gönderilemedi', 'error');
        } finally {
            // Input'u tekrar aktif et
            messageInput.disabled = false;
            messageInput.focus();
        }
    }

    startPolling() {
        // Her 3 saniyede bir yeni mesajları kontrol et
        this.messagePollingInterval = setInterval(() => {
            this.checkNewMessages();
        }, 3000);

        console.log('Polling başlatıldı (3 saniye)');
    }

    async checkNewMessages() {
        try {
            // Her zaman sohbet listesini güncelle
            await this.updateConversationList();

            // Aktif sohbet varsa mesajları kontrol et VE YÜKLE
            if (this.currentConversationId) {
                const response = await fetch(`api/get-messages.php?conversation_id=${this.currentConversationId}`);
                const result = await response.json();

                if (result.success && result.messages) {
                    // Mevcut mesaj sayısını al
                    const chatMessages = document.getElementById('chat-messages');
                    const currentMessageCount = chatMessages ? chatMessages.querySelectorAll('.message').length : 0;
                    const newMessageCount = result.messages.length;

                    console.log(`Mesaj kontrolü: Mevcut=${currentMessageCount}, Yeni=${newMessageCount}`);

                    // Yeni mesaj varsa VEYA mesaj sayısı farklıysa güncelle
                    if (newMessageCount !== currentMessageCount) {
                        console.log('Mesajlar güncelleniyor...');
                        this.displayMessages(result.messages);

                        // Son mesaj müşteriden mi kontrol et
                        if (newMessageCount > currentMessageCount) {
                            const lastMessage = result.messages[result.messages.length - 1];
                            if (lastMessage && lastMessage.sender_type === 'customer') {
                                // Bildirim göster
                                const conversation = result.conversation;
                                if (conversation && conversation.customer_name) {
                                    this.showNotification(
                                        conversation.customer_name,
                                        lastMessage.content
                                    );
                                }

                                // Ses çal
                                this.playNotificationSound();
                            }
                        }
                    }
                }
            }

        } catch (error) {
            console.error('Polling hatası:', error);
        }
    }

    async updateConversationList() {
        try {
            const response = await fetch('api/get-conversations.php');
            const result = await response.json();

            if (result.success && result.conversations) {
                // Her zaman sohbet listesini güncelle (son mesaj için)
                this.loadConversations();
            }
        } catch (error) {
            console.error('Sohbet listesi güncelleme hatası:', error);
        }
    }

    playNotificationSound() {
        // Basit bildirim sesi (opsiyonel)
        try {
            const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBTGH0fPTgjMGHm7A7+OZURE');
            audio.volume = 0.3;
            audio.play().catch(() => { });
        } catch (e) { }
    }

    startSSE() {
        // SSE devre dışı - Polling kullanılıyor
        console.log('SSE devre dışı, Polling aktif');
    }

    showNotification(title, message) {
        // Tarayıcı bildirimleri devre dışı - kullanıcı deneyimi için
        // Mesajlar zaten otomatik geliyor, bildirim popup'ı gereksiz
        console.log(`📬 Yeni mesaj: ${title} - ${message}`);

        // İsteyen kullanıcılar için (tarayıcı izni varsa sessizce gönder)
        if (Notification.permission === 'granted') {
            try {
                new Notification(title, {
                    body: message,
                    icon: 'https://profaj.co/multicrm/assets/images/logo.png',
                    tag: 'multicrm-notification'
                });
            } catch (e) {
                // Hata olursa sessizce devam et
            }
        }
    }

    stopSSE() {
        if (this.eventSource) {
            this.eventSource.close();
            this.eventSource = null;
        }
    }

    // Debounced update - birden fazla event'i topla ve bir kez update et
    scheduleUpdate(updateType) {
        this.pendingUpdates[updateType] = true;

        if (this.updateDebounce) {
            clearTimeout(this.updateDebounce);
        }

        // 500ms sonra topladığı update'leri yap
        this.updateDebounce = setTimeout(() => {
            if (this.pendingUpdates.messages) {
                if (this.currentConversationId) {
                    this.loadMessages(this.currentConversationId);
                }
                this.pendingUpdates.messages = false;
            }

            if (this.pendingUpdates.conversations) {
                this.loadConversations();
                this.pendingUpdates.conversations = false;
            }

            this.updateDebounce = null;
        }, 500);
    }

    async loadConversations() {
        try {
            const response = await fetch('api/get-conversations.php');
            const result = await response.json();

            if (result.success) {
                this.displayConversations(result.conversations);
            }
        } catch (error) {
            console.error('Sohbet yükleme hatası:', error);
        }
    }

    displayConversations(conversations) {
        const conversationsList = document.querySelector('.conversations-list');

        if (conversations.length === 0) {
            conversationsList.innerHTML = `
                <div class="no-conversations">
                    <i class="fas fa-comments"></i>
                    <p>Henüz sohbet yok</p>
                </div>
            `;
            return;
        }

        // Mevcut items'ları al
        const existingItems = conversationsList.querySelectorAll('.conversation-item');
        const existingIds = new Set([...existingItems].map(el => el.dataset.conversationId));
        const incomingIds = new Set(conversations.map(c => c.id));

        // Silinmiş olanları kaldır
        existingItems.forEach(item => {
            if (!incomingIds.has(item.dataset.conversationId)) {
                item.remove();
            }
        });

        // Yeni olanları ekle veya mevcut olanları güncelle
        conversations.forEach((conversation, index) => {
            const existingItem = conversationsList.querySelector(`[data-conversation-id="${conversation.id}"]`);

            if (existingItem) {
                // Zaten var, bilgileri güncelle (DOM yazma, sadece öznitelik değiştir)
                const infoDiv = existingItem.querySelector('.conversation-name');
                const msgDiv = existingItem.querySelector('.conversation-preview');
                const timeDiv = existingItem.querySelector('small');
                const badge = existingItem.querySelector('.unread-badge');

                if (infoDiv) infoDiv.textContent = this.escapeHtml(conversation.customer_name);
                if (msgDiv) msgDiv.textContent = this.escapeHtml(conversation.last_message || 'Henüz mesaj yok');
                if (timeDiv) timeDiv.textContent = conversation.last_message_time;

                if (conversation.unread_count > 0) {
                    if (badge) {
                        badge.textContent = conversation.unread_count;
                    } else {
                        const newBadge = document.createElement('span');
                        newBadge.className = 'unread-badge';
                        newBadge.textContent = conversation.unread_count;
                        existingItem.appendChild(newBadge);
                    }
                } else if (badge) {
                    badge.remove();
                }

                // Başa taşı (en yeni sohbet)
                conversationsList.insertBefore(existingItem, conversationsList.firstChild);
            } else {
                // Yeni item, ekle
                const conversationDiv = document.createElement('div');
                conversationDiv.className = 'conversation-item';
                conversationDiv.dataset.conversationId = conversation.id;

                // Platform ikonu belirle
                let platformIcon = 'fab fa-whatsapp';
                let platformClass = 'whatsapp';
                if (conversation.platform === 'facebook') {
                    platformIcon = 'fab fa-facebook-messenger';
                    platformClass = 'facebook';
                } else if (conversation.platform === 'instagram') {
                    platformIcon = 'fab fa-instagram';
                    platformClass = 'instagram';
                }

                conversationDiv.innerHTML = `
                    <div class="conversation-avatar ${platformClass}">
                        <i class="${platformIcon}"></i>
                    </div>
                    <div class="conversation-info">
                        <div class="conversation-name">${this.escapeHtml(conversation.customer_name)}</div>
                        <div class="conversation-preview">${this.escapeHtml(conversation.last_message || 'Henüz mesaj yok')}</div>
                        <small style="color: #a0aec0;">${conversation.last_message_time}</small>
                    </div>
                    ${conversation.unread_count > 0 ? `<span class="unread-badge">${conversation.unread_count}</span>` : ''}
                `;

                conversationsList.insertBefore(conversationDiv, conversationsList.firstChild);
            }
        });
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification ${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            ${message}
        `;

        document.body.appendChild(notification);

        // Göster
        setTimeout(() => notification.classList.add('show'), 100);

        // Gizle
        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    updateChatHeader(conversation) {
        // API'den gelen conversation object'i kullan
        if (conversation) {
            document.getElementById('chat-customer-name').textContent = conversation.customer_name || 'Bilinmeyen';
            document.getElementById('chat-customer-phone').textContent = conversation.customer_phone || '';

            // Update assignment dropdown if available
            if (typeof window.updateAssignmentDropdown === 'function') {
                window.updateAssignmentDropdown(conversation);
            }
        }

        // Butonları her zaman göster
        document.getElementById('customer-info-btn').style.display = 'inline-block';
        document.getElementById('delete-conversation-btn').style.display = 'inline-block';
    }
}

// Sohbet silme işlemleri
document.addEventListener('DOMContentLoaded', function () {
    // Sohbet silme butonu
    document.getElementById('delete-conversation-btn').addEventListener('click', function () {
        if (!window.app || !window.app.currentConversationId) {
            alert('Silinecek sohbet seçin');
            return;
        }

        // Modal'ı göster
        const deleteModal = new bootstrap.Modal(document.getElementById('deleteConversationModal'));
        deleteModal.show();
    });

    // Sohbet silme onayı
    document.getElementById('confirm-delete-conversation-btn').addEventListener('click', function () {
        if (!window.app || !window.app.currentConversationId) {
            return;
        }

        fetch('api/delete-conversation.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                conversation_id: window.app.currentConversationId
            })
        })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Modal'ı kapat
                    const deleteModal = bootstrap.Modal.getInstance(document.getElementById('deleteConversationModal'));
                    deleteModal.hide();

                    // Sayfayı yenile
                    location.reload();
                } else {
                    alert('Hata: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu');
            });
    });
});

// Chatbot yönetimi
MultiCRM.prototype.loadChatbots = async function () {
    try {
        const response = await fetch('api/chatbots/list.php');
        const data = await response.json();

        if (data.success && data.chatbots.length > 0) {
            const selector = document.getElementById('chatbot-selector');
            selector.innerHTML = '<option value="">Varsayılan Bot</option>';

            data.chatbots.forEach(bot => {
                const option = document.createElement('option');
                option.value = bot.id;
                option.textContent = bot.name;
                selector.appendChild(option);
            });

            // Chatbot değiştiğinde
            selector.addEventListener('change', async (e) => {
                if (!this.currentConversationId) return;

                const chatbotId = e.target.value || null;

                try {
                    const response = await fetch('api/chatbots/set-conversation-bot.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            conversation_id: this.currentConversationId,
                            chatbot_id: chatbotId
                        })
                    });

                    const result = await response.json();
                    if (result.success) {
                        this.showNotification(chatbotId ? 'Chatbot seçildi' : 'Varsayılan bot aktif', 'success');
                    }
                } catch (error) {
                    console.error('Chatbot seçilemedi:', error);
                }
            });
        }
    } catch (error) {
        console.error('Chatbot listesi yüklenemedi:', error);
    }
};

// Uygulama başlat
document.addEventListener('DOMContentLoaded', () => {
    window.app = new MultiCRM();
});

// Sayfa kapatılırken SSE bağlantısını kapat
window.addEventListener('beforeunload', () => {
    if (window.app && window.app.eventSource) {
        window.app.eventSource.close();
    }
});
