// Chatbot AI Yönetimi - JavaScript

let currentChatbotId = null;

// Chatbot seç
function selectChatbot(id) {
    currentChatbotId = id;

    // Aktif kartı işaretle
    document.querySelectorAll('.chatbot-card').forEach(card => {
        card.classList.remove('active');
    });
    document.querySelector(`[data-chatbot-id="${id}"]`).classList.add('active');

    // Detayları yükle
    loadChatbotDetails(id);
}

// Chatbot detaylarını yükle
async function loadChatbotDetails(id) {
    try {
        const response = await fetch(`api/chatbots/get.php?id=${id}`);
        const data = await response.json();

        if (data.success) {
            const bot = data.chatbot;

            // Detayları göster
            document.getElementById('empty-state').style.display = 'none';
            document.getElementById('chatbot-details').style.display = 'block';

            // Bilgileri doldur
            document.getElementById('chatbot-name').textContent = bot.name;
            document.getElementById('detail-name').textContent = bot.name;
            document.getElementById('detail-type').textContent = getTypeLabel(bot.type);
            document.getElementById('detail-description').textContent = bot.description || '-';
            document.getElementById('detail-prompt').value = bot.system_prompt;
            document.getElementById('detail-model').textContent = bot.model;
            document.getElementById('detail-temperature').textContent = bot.temperature;
            document.getElementById('detail-max-tokens').textContent = bot.max_tokens;

            // Zamanlama ayarlarını yükle
            const scheduleEnabled = document.getElementById('schedule-enabled');
            const scheduleSettings = document.getElementById('schedule-settings');

            scheduleEnabled.checked = bot.schedule_enabled == 1;
            scheduleSettings.style.display = bot.schedule_enabled == 1 ? 'block' : 'none';

            if (bot.schedule_start_time) {
                document.getElementById('schedule-start-time').value = bot.schedule_start_time;
            }
            if (bot.schedule_end_time) {
                document.getElementById('schedule-end-time').value = bot.schedule_end_time;
            }

            // Günleri yükle
            if (bot.schedule_days) {
                // Tüm gün checkbox'larını sıfırla
                document.querySelectorAll('.day-checkbox').forEach(checkbox => checkbox.checked = false);
                const days = bot.schedule_days.split(',');
                days.forEach(day => {
                    const checkbox = document.getElementById(`day-${day}`);
                    if (checkbox) checkbox.checked = true;
                });
            } else {
                // Eğer schedule_days yoksa tüm gün checkbox'larını sıfırla
                document.querySelectorAll('.day-checkbox').forEach(checkbox => checkbox.checked = false);
            }

            // Zamanlama toggle
            scheduleEnabled.onchange = function () {
                scheduleSettings.style.display = this.checked ? 'block' : 'none';
            };

            // Prompt değişikliğini dinle
            const promptTextarea = document.getElementById('detail-prompt');
            const saveBtn = document.getElementById('save-prompt-btn');
            let originalPrompt = bot.system_prompt;

            promptTextarea.oninput = function () {
                if (this.value !== originalPrompt) {
                    saveBtn.style.display = 'inline-block';
                } else {
                    saveBtn.style.display = 'none';
                }
            };

            // Bilgi bankasını yükle
            loadKnowledgeFiles(id);
        }
    } catch (error) {
        console.error('Chatbot detayları yüklenemedi:', error);
    }
}

// Bilgi bankası dosyalarını yükle
async function loadKnowledgeFiles(chatbotId) {
    try {
        const response = await fetch(`api/chatbots/get-knowledge.php?chatbot_id=${chatbotId}`);
        const data = await response.json();

        const container = document.getElementById('knowledge-files-list');
        container.innerHTML = '';

        if (data.success && data.files.length > 0) {
            data.files.forEach(file => {
                const fileDiv = document.createElement('div');
                fileDiv.className = 'knowledge-file';
                fileDiv.innerHTML = `
                    <div>
                        <i class="fas fa-file-${getFileIcon(file.file_type)} text-primary"></i>
                        <strong>${file.file_name}</strong>
                        <small class="text-muted">(${formatFileSize(file.file_size)})</small>
                    </div>
                    <button class="btn btn-sm btn-outline-danger" onclick="deleteKnowledgeFile(${file.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                `;
                container.appendChild(fileDiv);
            });
        } else {
            container.innerHTML = '<p class="text-muted text-center py-3">Henüz dosya yüklenmemiş</p>';
        }
    } catch (error) {
        console.error('Dosyalar yüklenemedi:', error);
    }
}

// Yeni chatbot oluştur
document.getElementById('createChatbotForm').addEventListener('submit', async (e) => {
    e.preventDefault();

    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData);

    try {
        const response = await fetch('api/chatbots/create.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });

        const result = await response.json();

        if (result.success) {
            bootstrap.Modal.getInstance(document.getElementById('createChatbotModal')).hide();
            location.reload();
        } else {
            alert('Hata: ' + result.error);
        }
    } catch (error) {
        alert('Bir hata oluştu');
    }
});

// Dosya yükleme
const uploadArea = document.getElementById('file-upload-area');
const fileInput = document.getElementById('knowledge-file-input');

uploadArea.addEventListener('click', () => fileInput.click());

uploadArea.addEventListener('dragover', (e) => {
    e.preventDefault();
    uploadArea.classList.add('dragover');
});

uploadArea.addEventListener('dragleave', () => {
    uploadArea.classList.remove('dragover');
});

uploadArea.addEventListener('drop', (e) => {
    e.preventDefault();
    uploadArea.classList.remove('dragover');
    handleFiles(e.dataTransfer.files);
});

fileInput.addEventListener('change', (e) => {
    handleFiles(e.target.files);
});

async function handleFiles(files) {
    if (!currentChatbotId) {
        alert('Önce bir chatbot seçin');
        return;
    }

    for (let file of files) {
        // Dosya tipi kontrolü
        const ext = file.name.split('.').pop().toLowerCase();
        if (!['pdf', 'txt', 'docx'].includes(ext)) {
            alert(`${file.name}: Desteklenmeyen dosya tipi`);
            continue;
        }

        // Boyut kontrolü (10MB)
        if (file.size > 10 * 1024 * 1024) {
            alert(`${file.name}: Dosya çok büyük (max 10MB)`);
            continue;
        }

        await uploadKnowledgeFile(file);
    }

    fileInput.value = '';
    loadKnowledgeFiles(currentChatbotId);
}

async function uploadKnowledgeFile(file) {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('chatbot_id', currentChatbotId);

    try {
        const response = await fetch('api/chatbots/upload-knowledge.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (!result.success) {
            alert(`${file.name}: ${result.error}`);
        }
    } catch (error) {
        alert(`${file.name}: Yükleme hatası`);
    }
}

async function deleteKnowledgeFile(id) {
    if (!confirm('Bu dosyayı silmek istediğinizden emin misiniz?')) return;

    try {
        const response = await fetch('api/chatbots/delete-knowledge.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ id })
        });

        const result = await response.json();

        if (result.success) {
            loadKnowledgeFiles(currentChatbotId);
        } else {
            alert('Hata: ' + result.error);
        }
    } catch (error) {
        alert('Bir hata oluştu');
    }
}

async function deleteChatbot() {
    if (!currentChatbotId) return;
    if (!confirm('Bu chatbot\'u silmek istediğinizden emin misiniz? Tüm bilgi bankası da silinecek!')) return;

    try {
        const response = await fetch('api/chatbots/delete.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ id: currentChatbotId })
        });

        const result = await response.json();

        if (result.success) {
            location.reload();
        } else {
            alert('Hata: ' + result.error);
        }
    } catch (error) {
        alert('Bir hata oluştu');
    }
}

// Sistem talimatını kaydet
async function saveSystemPrompt() {
    if (!currentChatbotId) return;

    const newPrompt = document.getElementById('detail-prompt').value;
    const saveBtn = document.getElementById('save-prompt-btn');

    saveBtn.disabled = true;
    saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Kaydediliyor...';

    try {
        const response = await fetch('api/chatbots/update-prompt.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                id: currentChatbotId,
                system_prompt: newPrompt
            })
        });

        const result = await response.json();

        if (result.success) {
            saveBtn.style.display = 'none';
            saveBtn.innerHTML = '<i class="fas fa-save"></i> Kaydet';
            saveBtn.disabled = false;

            // Orijinal prompt'u güncelle
            const promptTextarea = document.getElementById('detail-prompt');
            promptTextarea.oninput();

            alert('Sistem talimatı kaydedildi!');
        } else {
            alert('Hata: ' + result.error);
            saveBtn.disabled = false;
            saveBtn.innerHTML = '<i class="fas fa-save"></i> Kaydet';
        }
    } catch (error) {
        alert('Bir hata oluştu');
        saveBtn.disabled = false;
        saveBtn.innerHTML = '<i class="fas fa-save"></i> Kaydet';
    }
}

// Zamanlama ayarlarını kaydet
async function saveSchedule() {
    if (!currentChatbotId) return;

    const scheduleEnabled = document.getElementById('schedule-enabled').checked;
    const startTime = document.getElementById('schedule-start-time').value;
    const endTime = document.getElementById('schedule-end-time').value;

    // Seçili günleri topla
    const selectedDays = [];
    for (let i = 0; i <= 6; i++) {
        const checkbox = document.getElementById(`day-${i}`);
        if (checkbox && checkbox.checked) {
            selectedDays.push(i);
        }
    }

    if (scheduleEnabled && (!startTime || !endTime || selectedDays.length === 0)) {
        alert('Lütfen başlangıç saati, bitiş saati ve en az bir gün seçin!');
        return;
    }

    try {
        const response = await fetch('api/chatbots/save-schedule.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                id: currentChatbotId,
                schedule_enabled: scheduleEnabled,
                schedule_start_time: startTime,
                schedule_end_time: endTime,
                schedule_days: selectedDays.join(',')
            })
        });

        const result = await response.json();

        if (result.success) {
            alert('Zamanlama ayarları kaydedildi!');
        } else {
            alert('Hata: ' + result.error);
        }
    } catch (error) {
        alert('Bir hata oluştu');
    }
}

// Chatbot aktif/pasif durumunu değiştir
async function toggleChatbotStatus(id, activate) {
    try {
        const response = await fetch('api/chatbots/toggle-status.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                id: id,
                is_active: activate
            })
        });

        const result = await response.json();

        if (result.success) {
            // Sayfayı yenile - tüm sohbetlerdeki AI durumu güncellensin
            location.reload();
        } else {
            alert('Hata: ' + result.error);
        }
    } catch (error) {
        alert('Bir hata oluştu');
    }
}

// Yardımcı fonksiyonlar
function getTypeLabel(type) {
    const types = {
        'sales': 'Satış',
        'support': 'Teknik Destek',
        'info': 'Bilgi',
        'custom': 'Özel'
    };
    return types[type] || type;
}

function getFileIcon(type) {
    const icons = {
        'pdf': 'pdf',
        'txt': 'alt',
        'docx': 'word'
    };
    return icons[type] || 'file';
}

function formatFileSize(bytes) {
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
    return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
}
