<?php
/**
 * Instagram Webhook Diagnostics
 * Check Instagram webhook configuration and logs
 */

require_once '../config/database.php';
require_once '../includes/functions.php';

session_start();
if (!isset($_SESSION['user_id'])) {
    die('Unauthorized');
}

$user = getUserById($_SESSION['user_id']);
if (!in_array($user['role'], ['admin', 'supervisor'])) {
    die('Permission denied');
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instagram Webhook Diagnostics</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { padding: 20px; font-family: monospace; background: #f5f7fa; }
        .success { color: #22c55e; }
        .error { color: #ef4444; }
        .warning { color: #f59e0b; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 6px; max-height: 400px; overflow-y: auto; }
        .section { margin: 20px 0; padding: 20px; background: white; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #E4405F; }
        .gradient-header { background: linear-gradient(135deg, #f09433 0%,#e6683c 25%,#dc2743 50%,#cc2366 75%,#bc1888 100%); color: white; padding: 20px; border-radius: 8px; margin-bottom: 20px; }
    </style>
</head>
<body>
    <div class="gradient-header">
        <h1><i class="fab fa-instagram"></i> Instagram Webhook Diagnostics</h1>
    </div>

    <div class="section">
        <h3>1. Webhook URL</h3>
        <pre><?php echo 'https://profaj.co/multicrm/api/webhook-instagram.php'; ?></pre>
        <p><strong>Test URL:</strong></p>
        <pre><?php 
            $verifyToken = $instagram_config['verify_token'] ?? 'multicrm_facebook_verify_2024';
            echo "https://profaj.co/multicrm/api/webhook-instagram.php?hub.mode=subscribe&hub.verify_token=$verifyToken&hub.challenge=test123"; 
        ?></pre>
        <p>Bu URL'yi tarayıcıda açın. "test123" döndürürse webhook çalışıyor demektir.</p>
    </div>

    <div class="section">
        <h3>2. Instagram Configuration</h3>
        <?php if (isset($instagram_config)): ?>
            <p class="success">✅ Instagram config loaded</p>
            <pre><?php echo json_encode($instagram_config, JSON_PRETTY_PRINT); ?></pre>
        <?php else: ?>
            <p class="error">❌ Instagram config NOT loaded</p>
        <?php endif; ?>
    </div>

    <div class="section">
        <h3>3. Database Settings</h3>
        <?php
        try {
            $stmt = $pdo->query("SELECT * FROM settings WHERE `key` LIKE 'instagram_%'");
            $settings = $stmt->fetchAll();
            
            if (count($settings) > 0) {
                echo '<p class="success">✅ Instagram settings in database: ' . count($settings) . '</p>';
                echo '<table class="table table-sm">';
                echo '<tr><th>Key</th><th>Value</th></tr>';
                foreach ($settings as $setting) {
                    echo '<tr>';
                    echo '<td>' . $setting['key'] . '</td>';
                    echo '<td>' . (strlen($setting['value']) > 50 ? substr($setting['value'], 0, 50) . '...' : $setting['value']) . '</td>';
                    echo '</tr>';
                }
                echo '</table>';
            } else {
                echo '<p class="warning">⚠️ No Instagram settings in database</p>';
            }
        } catch (Exception $e) {
            echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
        }
        ?>
    </div>

    <div class="section">
        <h3>4. Recent Webhook Logs</h3>
        <?php
        $logFile = '../logs/instagram_webhook_' . date('Y-m-d') . '.log';
        if (file_exists($logFile)) {
            $logs = file_get_contents($logFile);
            $lines = explode("\n", $logs);
            $recent = array_slice(array_reverse($lines), 0, 50);
            echo '<p class="success">✅ Log file exists: ' . basename($logFile) . '</p>';
            echo '<pre>';
            echo htmlspecialchars(implode("\n", array_reverse($recent)));
            echo '</pre>';
        } else {
            echo '<p class="warning">⚠️ No log file for today</p>';
            echo '<p>Expected: ' . $logFile . '</p>';
            echo '<p class="info">Bu, Instagram\'dan hiç mesaj gelmediği anlamına gelir.</p>';
        }
        ?>
    </div>

    <div class="section">
        <h3>5. Recent Instagram Customers</h3>
        <?php
        try {
            $stmt = $pdo->query("SELECT * FROM customers WHERE instagram_id IS NOT NULL ORDER BY created_at DESC LIMIT 10");
            $customers = $stmt->fetchAll();
            
            if (count($customers) > 0) {
                echo '<p class="success">✅ Found ' . count($customers) . ' Instagram customers</p>';
                echo '<table class="table table-sm">';
                echo '<tr><th>ID</th><th>Name</th><th>Instagram ID</th><th>Created</th></tr>';
                foreach ($customers as $c) {
                    echo '<tr>';
                    echo '<td>' . $c['id'] . '</td>';
                    echo '<td>' . htmlspecialchars($c['name']) . '</td>';
                    echo '<td>' . $c['instagram_id'] . '</td>';
                    echo '<td>' . $c['created_at'] . '</td>';
                    echo '</tr>';
                }
                echo '</table>';
            } else {
                echo '<p class="warning">⚠️ No Instagram customers found</p>';
                echo '<p class="info">Instagram\'dan henüz mesaj gelmemiş.</p>';
            }
        } catch (Exception $e) {
            echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
        }
        ?>
    </div>

    <div class="section">
        <h3>6. Instagram Conversations</h3>
        <?php
        try {
            $stmt = $pdo->query("SELECT * FROM conversations WHERE platform = 'instagram' ORDER BY created_at DESC LIMIT 10");
            $conversations = $stmt->fetchAll();
            
            if (count($conversations) > 0) {
                echo '<p class="success">✅ Found ' . count($conversations) . ' Instagram conversations</p>';
                echo '<table class="table table-sm">';
                echo '<tr><th>ID</th><th>Customer ID</th><th>Status</th><th>Created</th></tr>';
                foreach ($conversations as $conv) {
                    echo '<tr>';
                    echo '<td>' . $conv['id'] . '</td>';
                    echo '<td>' . $conv['customer_id'] . '</td>';
                    echo '<td>' . $conv['status'] . '</td>';
                    echo '<td>' . $conv['created_at'] . '</td>';
                    echo '</tr>';
                }
                echo '</table>';
            } else {
                echo '<p class="warning">⚠️ No Instagram conversations found</p>';
            }
        } catch (Exception $e) {
            echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
        }
        ?>
    </div>

    <div class="section">
        <h3>7. Setup Checklist</h3>
        <div class="alert alert-info">
            <h5>Instagram DM için gerekli adımlar:</h5>
            <ol>
                <li><strong>Facebook Business hesabı</strong> - Instagram Business hesabınız Facebook Page'e bağlı olmalı</li>
                <li><strong>Facebook App</strong> - Instagram Messaging API eklenmiş olmalı</li>
                <li><strong>Permissions</strong> - <code>instagram_manage_messages</code> izni gerekli</li>
                <li><strong>Webhook Subscribe</strong> - Instagram → Settings → Webhooks → Subscribe to messages</li>
                <li><strong>Page Access Token</strong> - Facebook ayarlarındaki token Instagram için de geçerli</li>
            </ol>
        </div>
    </div>

    <p style="margin-top: 40px;">
        <a href="../index.php" class="btn btn-secondary">← Back to Dashboard</a>
        <a href="../settings.php" class="btn btn-primary">Settings</a>
        <a href="facebook-webhook-diagnostics.php" class="btn btn-info">Facebook Diagnostics</a>
    </p>
</body>
</html>
