<?php
/**
 * Assignment Engine - Automatic Conversation Assignment System
 * Handles round-robin, load-balanced, and rule-based assignment
 */

require_once __DIR__ . '/../config/database.php';

class AssignmentEngine {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Auto-assign conversation to best available agent
     */
    public function autoAssignConversation($conversation_id, $platform = 'whatsapp') {
        try {
            // Get assignment rules
            $rule = $this->getActiveAssignmentRule($platform);
            
            if (!$rule) {
                $this->log("No active assignment rule found for platform: $platform", 'WARNING');
                return false;
            }
            
            // Get available agents
            $agents = $this->getAvailableAgents($platform, $rule);
            
            if (empty($agents)) {
                $this->log("No available agents found for platform: $platform", 'WARNING');
                return false;
            }
            
            // Select agent based on assignment type
            $selectedAgent = null;
            switch ($rule['assignment_type']) {
                case 'round_robin':
                    $selectedAgent = $this->roundRobinAssignment($agents);
                    break;
                case 'load_balanced':
                    $selectedAgent = $this->loadBalancedAssignment($agents);
                    break;
                default:
                    $selectedAgent = $agents[0]; // Fallback to first agent
            }
            
            if (!$selectedAgent) {
                $this->log("Failed to select agent for conversation: $conversation_id", 'ERROR');
                return false;
            }
            
            // Assign conversation
            $success = $this->assignConversation($conversation_id, $selectedAgent['id'], $rule['assignment_type']);
            
            if ($success) {
                $this->log("Conversation $conversation_id assigned to agent {$selectedAgent['id']} ({$selectedAgent['name']}) using {$rule['assignment_type']}");
                
                // Update agent availability
                $this->updateAgentWorkload($selectedAgent['id']);
                
                // Create notification
                $this->createAssignmentNotification($conversation_id, $selectedAgent['id']);
            }
            
            return $success;
            
        } catch (Exception $e) {
            $this->log("Auto-assignment error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Get active assignment rule for platform
     */
    private function getActiveAssignmentRule($platform) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM assignment_rules 
            WHERE (platform = ? OR platform = 'all') 
            AND is_active = 1 
            ORDER BY priority DESC 
            LIMIT 1
        ");
        $stmt->execute([$platform]);
        return $stmt->fetch();
    }
    
    /**
     * Get available agents based on rules
     */
    public function getAvailableAgents($platform = null, $rule = null) {
        $maxConversations = $rule['max_conversations_per_agent'] ?? 10;
        
        $sql = "
            SELECT u.id, u.name, u.email, u.role,
                   aa.status, aa.current_conversations, aa.max_conversations,
                   COUNT(c.id) as active_conversations
            FROM users u
            INNER JOIN agent_availability aa ON u.id = aa.user_id
            LEFT JOIN conversations c ON c.assigned_to = u.id AND c.status = 'active'
            WHERE u.status = 'active' 
            AND u.role IN ('agent', 'admin', 'supervisor')
            AND aa.status = 'online'
            AND aa.auto_accept = 1
        ";
        
        // Add platform permission check if needed
        if ($platform) {
            $sql .= " AND (
                NOT EXISTS (
                    SELECT 1 FROM user_permissions up 
                    WHERE up.user_id = u.id 
                    AND up.permission = 'platform_access' 
                    AND up.resource != ?
                )
                OR EXISTS (
                    SELECT 1 FROM user_permissions up 
                    WHERE up.user_id = u.id 
                    AND up.permission = 'platform_access' 
                    AND up.resource = ?
                )
            )";
        }
        
        $sql .= "
            GROUP BY u.id
            HAVING active_conversations < ?
            ORDER BY active_conversations ASC, u.id ASC
        ";
        
        $stmt = $this->pdo->prepare($sql);
        
        if ($platform) {
            $stmt->execute([$platform, $platform, $maxConversations]);
        } else {
            $stmt->execute([$maxConversations]);
        }
        
        return $stmt->fetchAll();
    }
    
    /**
     * Round-robin assignment - rotate through agents
     */
    private function roundRobinAssignment($agents) {
        // Get last assigned agent
        $stmt = $this->pdo->prepare("
            SELECT assigned_to FROM conversations 
            WHERE assignment_type = 'round_robin' 
            ORDER BY assigned_at DESC 
            LIMIT 1
        ");
        $stmt->execute();
        $lastAssigned = $stmt->fetchColumn();
        
        // Find next agent in rotation
        if ($lastAssigned) {
            $currentIndex = array_search($lastAssigned, array_column($agents, 'id'));
            $nextIndex = ($currentIndex + 1) % count($agents);
            return $agents[$nextIndex];
        }
        
        // No previous assignment, return first agent
        return $agents[0];
    }
    
    /**
     * Load-balanced assignment - assign to agent with least conversations
     */
    private function loadBalancedAssignment($agents) {
        // Agents are already sorted by active_conversations ASC
        return $agents[0];
    }
    
    /**
     * Assign conversation to specific agent
     */
    public function assignConversation($conversation_id, $agent_id, $assignment_type = 'manual') {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE conversations 
                SET assigned_to = ?, 
                    assignment_type = ?, 
                    assigned_at = NOW(),
                    updated_at = NOW()
                WHERE id = ?
            ");
            
            $success = $stmt->execute([$agent_id, $assignment_type, $conversation_id]);
            
            if ($success) {
                // Log activity
                $this->logActivity($agent_id, 'conversation_assigned', 'conversation', $conversation_id, [
                    'assignment_type' => $assignment_type
                ]);
            }
            
            return $success;
            
        } catch (Exception $e) {
            $this->log("Assignment error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Transfer conversation to another agent
     */
    public function transferConversation($conversation_id, $to_agent_id, $from_agent_id, $transferred_by, $reason = null, $notes = null) {
        try {
            $this->pdo->beginTransaction();
            
            // Update conversation
            $stmt = $this->pdo->prepare("
                UPDATE conversations 
                SET assigned_to = ?, 
                    assignment_type = 'manual',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$to_agent_id, $conversation_id]);
            
            // Record transfer
            $stmt = $this->pdo->prepare("
                INSERT INTO conversation_transfers 
                (conversation_id, from_agent_id, to_agent_id, transferred_by, reason, notes) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$conversation_id, $from_agent_id, $to_agent_id, $transferred_by, $reason, $notes]);
            
            // Log activity
            $this->logActivity($transferred_by, 'conversation_transferred', 'conversation', $conversation_id, [
                'from_agent' => $from_agent_id,
                'to_agent' => $to_agent_id,
                'reason' => $reason
            ]);
            
            // Update agent workloads
            $this->updateAgentWorkload($from_agent_id);
            $this->updateAgentWorkload($to_agent_id);
            
            // Create notification for new agent
            $this->createTransferNotification($conversation_id, $to_agent_id, $from_agent_id);
            
            $this->pdo->commit();
            return true;
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            $this->log("Transfer error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Update agent's current workload
     */
    private function updateAgentWorkload($agent_id) {
        $stmt = $this->pdo->prepare("
            UPDATE agent_availability 
            SET current_conversations = (
                SELECT COUNT(*) FROM conversations 
                WHERE assigned_to = ? AND status = 'active'
            ),
            last_activity_at = NOW()
            WHERE user_id = ?
        ");
        return $stmt->execute([$agent_id, $agent_id]);
    }
    
    /**
     * Get agent's current workload
     */
    public function getAgentWorkload($agent_id) {
        $stmt = $this->pdo->prepare("
            SELECT 
                aa.current_conversations,
                aa.max_conversations,
                aa.status,
                COUNT(c.id) as actual_conversations
            FROM agent_availability aa
            LEFT JOIN conversations c ON c.assigned_to = aa.user_id AND c.status = 'active'
            WHERE aa.user_id = ?
            GROUP BY aa.user_id
        ");
        $stmt->execute([$agent_id]);
        return $stmt->fetch();
    }
    
    /**
     * Get unassigned conversations
     */
    public function getUnassignedConversations($platform = null, $limit = 50) {
        $sql = "
            SELECT c.*, cu.name as customer_name, cu.phone, cu.facebook_id
            FROM conversations c
            LEFT JOIN customers cu ON c.customer_id = cu.id
            WHERE c.assigned_to IS NULL OR c.assigned_to = 0
            AND c.status = 'active'
        ";
        
        if ($platform) {
            $sql .= " AND c.platform = ?";
        }
        
        $sql .= " ORDER BY c.created_at ASC LIMIT ?";
        
        $stmt = $this->pdo->prepare($sql);
        
        if ($platform) {
            $stmt->execute([$platform, $limit]);
        } else {
            $stmt->execute([$limit]);
        }
        
        return $stmt->fetchAll();
    }
    
    /**
     * Create assignment notification
     */
    private function createAssignmentNotification($conversation_id, $agent_id) {
        try {
            // Get conversation details
            $stmt = $this->pdo->prepare("
                SELECT c.*, cu.name as customer_name 
                FROM conversations c 
                LEFT JOIN customers cu ON c.customer_id = cu.id 
                WHERE c.id = ?
            ");
            $stmt->execute([$conversation_id]);
            $conversation = $stmt->fetch();
            
            if ($conversation) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO notifications 
                    (user_id, conversation_id, type, title, message) 
                    VALUES (?, ?, 'new_conversation', ?, ?)
                ");
                
                $title = "Yeni Sohbet Atandı";
                $message = "{$conversation['customer_name']} ile yeni bir sohbet size atandı.";
                
                $stmt->execute([$agent_id, $conversation_id, $title, $message]);
            }
        } catch (Exception $e) {
            $this->log("Notification error: " . $e->getMessage(), 'ERROR');
        }
    }
    
    /**
     * Create transfer notification
     */
    private function createTransferNotification($conversation_id, $to_agent_id, $from_agent_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT c.*, cu.name as customer_name, u.name as from_agent_name
                FROM conversations c 
                LEFT JOIN customers cu ON c.customer_id = cu.id 
                LEFT JOIN users u ON u.id = ?
                WHERE c.id = ?
            ");
            $stmt->execute([$from_agent_id, $conversation_id]);
            $data = $stmt->fetch();
            
            if ($data) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO notifications 
                    (user_id, conversation_id, type, title, message) 
                    VALUES (?, ?, 'new_conversation', ?, ?)
                ");
                
                $title = "Sohbet Transfer Edildi";
                $message = "{$data['from_agent_name']} tarafından {$data['customer_name']} sohbeti size transfer edildi.";
                
                $stmt->execute([$to_agent_id, $conversation_id, $title, $message]);
            }
        } catch (Exception $e) {
            $this->log("Transfer notification error: " . $e->getMessage(), 'ERROR');
        }
    }
    
    /**
     * Log activity
     */
    private function logActivity($user_id, $action, $resource_type, $resource_id, $details = []) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO activity_logs 
                (user_id, action, resource_type, resource_id, details, ip_address, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $ip = $_SERVER['REMOTE_ADDR'] ?? null;
            $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
            
            $stmt->execute([
                $user_id,
                $action,
                $resource_type,
                $resource_id,
                json_encode($details),
                $ip,
                $userAgent
            ]);
        } catch (Exception $e) {
            $this->log("Activity log error: " . $e->getMessage(), 'ERROR');
        }
    }
    
    /**
     * Log message
     */
    private function log($message, $level = 'INFO') {
        $logFile = __DIR__ . '/../logs/assignment_' . date('Y-m-d') . '.log';
        $logMessage = '[' . date('Y-m-d H:i:s') . '] [' . $level . '] ' . $message . PHP_EOL;
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
}

// Helper function to get assignment engine instance
function getAssignmentEngine() {
    global $pdo;
    return new AssignmentEngine($pdo);
}
?>
