<?php
/**
 * Manual Table Creation - Step by Step
 * This creates missing tables one by one with error handling
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../config/database.php';

echo "<!DOCTYPE html>
<html lang='tr'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Manual Table Creation - MultiCRM</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            max-width: 900px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f7fa;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        h1 { color: #1f93ff; margin-top: 0; }
        .success { background: #dcfce7; color: #166534; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .error { background: #fee2e2; color: #991b1b; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .warning { background: #fef3c7; color: #92400e; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .info { background: #dbeafe; color: #1e40af; padding: 12px; border-radius: 6px; margin: 10px 0; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 6px; overflow-x: auto; font-size: 12px; }
        .btn { background: #1f93ff; color: white; padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block; margin-top: 20px; }
        .btn:hover { background: #0066cc; }
    </style>
</head>
<body>
    <div class='container'>";

echo "<h1>🔧 Manual Table Creation</h1>";

$tables = [
    'assignment_rules' => "CREATE TABLE IF NOT EXISTS `assignment_rules` (
        `id` INT(11) NOT NULL AUTO_INCREMENT,
        `name` VARCHAR(255) NOT NULL,
        `description` TEXT DEFAULT NULL,
        `platform` ENUM('whatsapp', 'facebook', 'instagram', 'all') DEFAULT 'all',
        `assignment_type` ENUM('round_robin', 'load_balanced', 'manual', 'skill_based') DEFAULT 'round_robin',
        `priority` INT DEFAULT 0,
        `conditions` JSON DEFAULT NULL,
        `target_agents` JSON DEFAULT NULL,
        `max_conversations_per_agent` INT DEFAULT 10,
        `working_hours_only` TINYINT(1) DEFAULT 1,
        `is_active` TINYINT(1) DEFAULT 1,
        `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
        `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        KEY `idx_platform` (`platform`),
        KEY `idx_is_active` (`is_active`),
        KEY `idx_priority` (`priority`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
    
    'user_permissions' => "CREATE TABLE IF NOT EXISTS `user_permissions` (
        `id` INT(11) NOT NULL AUTO_INCREMENT,
        `user_id` INT(11) NOT NULL,
        `permission` VARCHAR(100) NOT NULL,
        `resource` VARCHAR(100) DEFAULT NULL,
        `granted_by` INT(11) DEFAULT NULL,
        `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        UNIQUE KEY `idx_user_permission` (`user_id`, `permission`, `resource`),
        KEY `idx_permission` (`permission`),
        FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
        FOREIGN KEY (`granted_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
    
    'conversation_transfers' => "CREATE TABLE IF NOT EXISTS `conversation_transfers` (
        `id` INT(11) NOT NULL AUTO_INCREMENT,
        `conversation_id` INT(11) NOT NULL,
        `from_agent_id` INT(11) NOT NULL,
        `to_agent_id` INT(11) NOT NULL,
        `transferred_by` INT(11) DEFAULT NULL,
        `reason` TEXT DEFAULT NULL,
        `notes` TEXT DEFAULT NULL,
        `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        KEY `idx_conversation_id` (`conversation_id`),
        KEY `idx_from_agent` (`from_agent_id`),
        KEY `idx_to_agent` (`to_agent_id`),
        KEY `idx_created_at` (`created_at`),
        FOREIGN KEY (`conversation_id`) REFERENCES `conversations` (`id`) ON DELETE CASCADE,
        FOREIGN KEY (`from_agent_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
        FOREIGN KEY (`to_agent_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
        FOREIGN KEY (`transferred_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
    
    'agent_availability' => "CREATE TABLE IF NOT EXISTS `agent_availability` (
        `id` INT(11) NOT NULL AUTO_INCREMENT,
        `user_id` INT(11) NOT NULL,
        `status` ENUM('online', 'offline', 'away', 'busy') DEFAULT 'offline',
        `max_conversations` INT DEFAULT 10,
        `current_conversations` INT DEFAULT 0,
        `auto_accept` TINYINT(1) DEFAULT 1,
        `working_hours` JSON DEFAULT NULL,
        `last_activity_at` TIMESTAMP NULL DEFAULT NULL,
        `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
        `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        UNIQUE KEY `idx_user_id` (`user_id`),
        KEY `idx_status` (`status`),
        FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
    
    'agent_statistics' => "CREATE TABLE IF NOT EXISTS `agent_statistics` (
        `id` INT(11) NOT NULL AUTO_INCREMENT,
        `user_id` INT(11) NOT NULL,
        `date` DATE NOT NULL,
        `platform` ENUM('whatsapp', 'facebook', 'instagram', 'all') DEFAULT 'all',
        `conversations_handled` INT DEFAULT 0,
        `messages_sent` INT DEFAULT 0,
        `messages_received` INT DEFAULT 0,
        `avg_first_response_time` INT DEFAULT NULL,
        `avg_response_time` INT DEFAULT NULL,
        `conversations_resolved` INT DEFAULT 0,
        `conversations_transferred` INT DEFAULT 0,
        `customer_satisfaction_score` DECIMAL(3,2) DEFAULT NULL,
        `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
        `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        UNIQUE KEY `idx_user_date_platform` (`user_id`, `date`, `platform`),
        KEY `idx_date` (`date`),
        KEY `idx_platform` (`platform`),
        FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci"
];

$successCount = 0;
$errorCount = 0;

foreach ($tables as $tableName => $sql) {
    echo "<h3>Creating table: $tableName</h3>";
    
    try {
        $pdo->exec($sql);
        echo "<div class='success'>✅ Table `$tableName` created successfully</div>";
        $successCount++;
    } catch (PDOException $e) {
        $errorMsg = $e->getMessage();
        
        if (strpos($errorMsg, 'already exists') !== false) {
            echo "<div class='warning'>⚠️ Table `$tableName` already exists (skipped)</div>";
            $successCount++;
        } else {
            echo "<div class='error'>❌ Error creating `$tableName`:<br>" . htmlspecialchars($errorMsg) . "</div>";
            $errorCount++;
        }
    }
}

// Insert default data
echo "<h2>Inserting Default Data</h2>";

try {
    // Default assignment rule
    $pdo->exec("INSERT IGNORE INTO `assignment_rules` (`name`, `description`, `platform`, `assignment_type`, `priority`, `is_active`) 
                VALUES ('Default Auto Assignment', 'Automatically assign new conversations using round-robin', 'all', 'round_robin', 1, 1)");
    echo "<div class='success'>✅ Default assignment rule created</div>";
} catch (PDOException $e) {
    echo "<div class='warning'>⚠️ Assignment rule: " . htmlspecialchars($e->getMessage()) . "</div>";
}

try {
    // Create availability records for existing users
    $pdo->exec("INSERT IGNORE INTO `agent_availability` (`user_id`, `status`, `max_conversations`, `auto_accept`)
                SELECT `id`, 'online', 10, 1 FROM `users` WHERE `status` = 'active'");
    echo "<div class='success'>✅ Agent availability records created</div>";
} catch (PDOException $e) {
    echo "<div class='warning'>⚠️ Availability: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Update columns in existing tables
echo "<h2>Updating Existing Tables</h2>";

$columnUpdates = [
    "ALTER TABLE `customers` ADD COLUMN IF NOT EXISTS `facebook_id` VARCHAR(255) DEFAULT NULL AFTER `email`",
    "ALTER TABLE `customers` ADD COLUMN IF NOT EXISTS `instagram_id` VARCHAR(255) DEFAULT NULL AFTER `facebook_id`",
    "ALTER TABLE `customers` ADD COLUMN IF NOT EXISTS `preferred_channel` ENUM('whatsapp', 'facebook', 'instagram', 'any') DEFAULT 'any' AFTER `instagram_id`",
    "ALTER TABLE `customers` ADD COLUMN IF NOT EXISTS `last_seen_at` TIMESTAMP NULL DEFAULT NULL AFTER `preferred_channel`",
    "ALTER TABLE `conversations` ADD COLUMN IF NOT EXISTS `platform` ENUM('whatsapp', 'facebook', 'instagram') DEFAULT 'whatsapp' AFTER `status`",
    "ALTER TABLE `conversations` ADD COLUMN IF NOT EXISTS `assignment_type` ENUM('manual', 'auto', 'round_robin', 'load_balanced') DEFAULT 'auto' AFTER `platform`",
    "ALTER TABLE `conversations` ADD COLUMN IF NOT EXISTS `assigned_at` TIMESTAMP NULL DEFAULT NULL AFTER `assignment_type`"
];

foreach ($columnUpdates as $sql) {
    try {
        $pdo->exec($sql);
        echo "<div class='success'>✅ Column added</div>";
    } catch (PDOException $e) {
        $msg = $e->getMessage();
        if (strpos($msg, 'Duplicate column') !== false || strpos($msg, 'check that it exists') !== false) {
            echo "<div class='warning'>⚠️ Column already exists (skipped)</div>";
        } else {
            echo "<div class='error'>❌ " . htmlspecialchars($msg) . "</div>";
        }
    }
}

// Final verification
echo "<h2>🔍 Final Verification</h2>";

$allTablesExist = true;
foreach (array_keys($tables) as $tableName) {
    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM `$tableName`");
        $count = $stmt->fetchColumn();
        echo "<div class='success'>✓ Table `$tableName` exists ($count rows)</div>";
    } catch (PDOException $e) {
        echo "<div class='error'>✗ Table `$tableName` NOT found</div>";
        $allTablesExist = false;
    }
}

if ($allTablesExist) {
    echo "<div class='success'><strong>🎉 All tables created successfully!</strong></div>";
    echo "<p><a href='../index.php' class='btn'>Go to MultiCRM Dashboard</a></p>";
} else {
    echo "<div class='error'><strong>❌ Some tables are still missing.</strong></div>";
    echo "<p>Please check the errors above and contact support if needed.</p>";
}

echo "</div></body></html>";
?>
