<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Veritabanı Güncelleme Sihirbazı - MultiCRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .wizard-container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            padding: 40px;
        }
        .step {
            display: none;
        }
        .step.active {
            display: block;
            animation: fadeIn 0.5s;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 40px;
        }
        .step-item {
            flex: 1;
            text-align: center;
            position: relative;
        }
        .step-item:not(:last-child)::after {
            content: '';
            position: absolute;
            top: 20px;
            left: 50%;
            width: 100%;
            height: 2px;
            background: #e0e0e0;
            z-index: -1;
        }
        .step-item.active .step-number {
            background: #667eea;
            color: white;
        }
        .step-item.completed .step-number {
            background: #28a745;
            color: white;
        }
        .step-number {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e0e0e0;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            margin-bottom: 10px;
        }
        .update-item {
            padding: 15px;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            margin-bottom: 10px;
        }
        .update-item.success {
            background: #d4edda;
            border-color: #c3e6cb;
        }
        .update-item.error {
            background: #f8d7da;
            border-color: #f5c6cb;
        }
        .update-item.pending {
            background: #fff3cd;
            border-color: #ffeaa7;
        }
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
        }
        .btn-primary:hover {
            background: linear-gradient(135deg, #5568d3 0%, #6a3f8f 100%);
        }
    </style>
</head>
<body>
    <div class="wizard-container">
        <h2 class="text-center mb-4">
            <i class="fas fa-database"></i> Veritabanı Güncelleme Sihirbazı
        </h2>
        
        <!-- Step Indicator -->
        <div class="step-indicator">
            <div class="step-item active" data-step="1">
                <div class="step-number">1</div>
                <small>Kontrol</small>
            </div>
            <div class="step-item" data-step="2">
                <div class="step-number">2</div>
                <small>Güncelleme</small>
            </div>
            <div class="step-item" data-step="3">
                <div class="step-number">3</div>
                <small>Tamamlandı</small>
            </div>
        </div>

        <!-- Step 1: Sistem Kontrolü -->
        <div class="step active" id="step-1">
            <h4><i class="fas fa-check-circle"></i> Sistem Kontrolü</h4>
            <p>Veritabanı durumu kontrol ediliyor...</p>
            
            <div id="check-results"></div>
            
            <div class="text-center mt-4">
                <button class="btn btn-primary" id="start-update-btn" disabled>
                    <i class="fas fa-arrow-right"></i> Güncellemeleri Başlat
                </button>
            </div>
        </div>

        <!-- Step 2: Güncelleme -->
        <div class="step" id="step-2">
            <h4><i class="fas fa-sync-alt"></i> Veritabanı Güncellemeleri</h4>
            <p>Gerekli güncellemeler uygulanıyor...</p>
            
            <div id="update-results"></div>
            
            <div class="text-center mt-4">
                <button class="btn btn-primary" id="finish-btn" disabled>
                    <i class="fas fa-check"></i> Tamamla
                </button>
            </div>
        </div>

        <!-- Step 3: Tamamlandı -->
        <div class="step" id="step-3">
            <div class="text-center">
                <i class="fas fa-check-circle text-success" style="font-size: 80px;"></i>
                <h3 class="mt-4">Güncelleme Tamamlandı!</h3>
                <p class="text-muted">Veritabanı başarıyla güncellendi.</p>
                
                <div class="alert alert-info mt-4">
                    <h5><i class="fas fa-info-circle"></i> Yapılan Güncellemeler:</h5>
                    <ul class="text-start" id="summary-list"></ul>
                </div>
                
                <div class="mt-4">
                    <a href="../index.php" class="btn btn-primary btn-lg">
                        <i class="fas fa-home"></i> Ana Sayfaya Git
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentStep = 1;
        let updateResults = [];

        // Sayfa yüklendiğinde kontrol başlat
        document.addEventListener('DOMContentLoaded', () => {
            checkDatabase();
        });

        // Kontrol butonları
        document.getElementById('start-update-btn').addEventListener('click', () => {
            showStep(2);
            applyUpdates();
        });

        document.getElementById('finish-btn').addEventListener('click', () => {
            showStep(3);
            showSummary();
        });

        function showStep(step) {
            // Mevcut adımı gizle
            document.querySelectorAll('.step').forEach(s => s.classList.remove('active'));
            document.querySelectorAll('.step-item').forEach(s => s.classList.remove('active'));
            
            // Yeni adımı göster
            document.getElementById(`step-${step}`).classList.add('active');
            document.querySelector(`.step-item[data-step="${step}"]`).classList.add('active');
            
            // Önceki adımları tamamlandı olarak işaretle
            for (let i = 1; i < step; i++) {
                document.querySelector(`.step-item[data-step="${i}"]`).classList.add('completed');
            }
            
            currentStep = step;
        }

        async function checkDatabase() {
            const resultsDiv = document.getElementById('check-results');
            resultsDiv.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin fa-2x"></i><p>Kontrol ediliyor...</p></div>';
            
            try {
                const response = await fetch('database-check-api.php');
                const result = await response.json();
                
                resultsDiv.innerHTML = '';
                
                if (result.checks) {
                    result.checks.forEach(check => {
                        const itemDiv = document.createElement('div');
                        itemDiv.className = `update-item ${check.status}`;
                        itemDiv.innerHTML = `
                            <div class="d-flex align-items-center">
                                <i class="fas ${check.status === 'success' ? 'fa-check-circle text-success' : 'fa-exclamation-circle text-warning'} me-3"></i>
                                <div class="flex-grow-1">
                                    <strong>${check.title}</strong>
                                    <p class="mb-0 small">${check.message}</p>
                                </div>
                            </div>
                        `;
                        resultsDiv.appendChild(itemDiv);
                    });
                    
                    // Güncelleme gerekiyorsa butonu aktif et
                    if (result.needs_update) {
                        document.getElementById('start-update-btn').disabled = false;
                    } else {
                        resultsDiv.innerHTML += `
                            <div class="alert alert-success mt-3">
                                <i class="fas fa-check-circle"></i> Veritabanı güncel! Güncelleme gerekmiyor.
                            </div>
                        `;
                    }
                }
                
            } catch (error) {
                resultsDiv.innerHTML = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle"></i> Kontrol sırasında hata oluştu: ${error.message}
                    </div>
                `;
            }
        }

        async function applyUpdates() {
            const resultsDiv = document.getElementById('update-results');
            resultsDiv.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin fa-2x"></i><p>Güncellemeler uygulanıyor...</p></div>';
            
            try {
                const response = await fetch('database-update-api.php');
                const result = await response.json();
                
                resultsDiv.innerHTML = '';
                updateResults = result.updates || [];
                
                if (result.updates) {
                    result.updates.forEach(update => {
                        const itemDiv = document.createElement('div');
                        itemDiv.className = `update-item ${update.success ? 'success' : 'error'}`;
                        itemDiv.innerHTML = `
                            <div class="d-flex align-items-center">
                                <i class="fas ${update.success ? 'fa-check-circle text-success' : 'fa-times-circle text-danger'} me-3"></i>
                                <div class="flex-grow-1">
                                    <strong>${update.title}</strong>
                                    <p class="mb-0 small">${update.message}</p>
                                </div>
                            </div>
                        `;
                        resultsDiv.appendChild(itemDiv);
                    });
                    
                    document.getElementById('finish-btn').disabled = false;
                }
                
            } catch (error) {
                resultsDiv.innerHTML = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle"></i> Güncelleme sırasında hata oluştu: ${error.message}
                    </div>
                `;
            }
        }

        function showSummary() {
            const summaryList = document.getElementById('summary-list');
            summaryList.innerHTML = '';
            
            const successUpdates = updateResults.filter(u => u.success);
            
            if (successUpdates.length > 0) {
                successUpdates.forEach(update => {
                    const li = document.createElement('li');
                    li.textContent = update.title;
                    summaryList.appendChild(li);
                });
            } else {
                summaryList.innerHTML = '<li>Güncelleme yapılmadı (veritabanı zaten güncel)</li>';
            }
        }
    </script>
</body>
</html>
