<?php
/**
 * Database Migration Runner
 * Run this file ONCE to update your database schema
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../config/database.php';

echo "<!DOCTYPE html>
<html lang='tr'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Database Migration - MultiCRM v2.0</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f7fa;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        h1 {
            color: #1f93ff;
            margin-top: 0;
        }
        .success {
            background: #dcfce7;
            color: #166534;
            padding: 12px;
            border-radius: 6px;
            margin: 10px 0;
        }
        .error {
            background: #fee2e2;
            color: #991b1b;
            padding: 12px;
            border-radius: 6px;
            margin: 10px 0;
        }
        .warning {
            background: #fef3c7;
            color: #92400e;
            padding: 12px;
            border-radius: 6px;
            margin: 10px 0;
        }
        .info {
            background: #dbeafe;
            color: #1e40af;
            padding: 12px;
            border-radius: 6px;
            margin: 10px 0;
        }
        pre {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 6px;
            overflow-x: auto;
            font-size: 13px;
        }
        .btn {
            background: #1f93ff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
        }
        .btn:hover {
            background: #0066cc;
        }
    </style>
</head>
<body>
    <div class='container'>";

echo "<h1>🚀 MultiCRM v2.0 - Database Migration</h1>";

try {
    // Read migration file
    $migrationFile = __DIR__ . '/database-migration.sql';
    
    if (!file_exists($migrationFile)) {
        throw new Exception("Migration file not found: $migrationFile");
    }
    
    $sql = file_get_contents($migrationFile);
    
    if ($sql === false) {
        throw new Exception("Failed to read migration file");
    }
    
    echo "<div class='info'><strong>📋 Migration file loaded successfully</strong></div>";
    
    // Split SQL into individual statements
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && !preg_match('/^--/', $stmt);
        }
    );
    
    echo "<div class='info'><strong>📊 Found " . count($statements) . " SQL statements to execute</strong></div>";
    
    // Execute each statement
    $successCount = 0;
    $errorCount = 0;
    $errors = [];
    
    foreach ($statements as $index => $statement) {
        try {
            $pdo->exec($statement);
            $successCount++;
        } catch (PDOException $e) {
            // Some errors are acceptable (e.g., column already exists)
            $errorMsg = $e->getMessage();
            if (
                strpos($errorMsg, 'Duplicate column') !== false ||
                strpos($errorMsg, 'Duplicate key') !== false ||
                strpos($errorMsg, 'already exists') !== false
            ) {
                // Acceptable error - column/table already exists
                $successCount++;
            } else {
                $errorCount++;
                $errors[] = [
                    'statement' => substr($statement, 0, 100) . '...',
                    'error' => $errorMsg
                ];
            }
        }
    }
    
    echo "<div class='success'><strong>✅ Migration completed!</strong><br>";
    echo "Successful: $successCount<br>";
    echo "Errors: $errorCount</div>";
    
    if (!empty($errors)) {
        echo "<div class='warning'><strong>⚠️ Some errors occurred:</strong></div>";
        foreach ($errors as $error) {
            echo "<div class='error'>";
            echo "<strong>Statement:</strong> " . htmlspecialchars($error['statement']) . "<br>";
            echo "<strong>Error:</strong> " . htmlspecialchars($error['error']);
            echo "</div>";
        }
    }
    
    // Verify critical tables
    echo "<h2>🔍 Verification</h2>";
    
    $criticalTables = [
        'assignment_rules',
        'user_permissions',
        'activity_logs',
        'conversation_transfers',
        'agent_availability',
        'agent_statistics'
    ];
    
    $allTablesExist = true;
    foreach ($criticalTables as $table) {
        try {
            $stmt = $pdo->query("SELECT COUNT(*) FROM `$table`");
            $count = $stmt->fetchColumn();
            echo "<div class='success'>✓ Table `$table` exists ($count rows)</div>";
        } catch (PDOException $e) {
            echo "<div class='error'>✗ Table `$table` NOT found</div>";
            $allTablesExist = false;
        }
    }
    
    if ($allTablesExist) {
        echo "<div class='success'><strong>🎉 All critical tables created successfully!</strong></div>";
        echo "<p><a href='../index.php' class='btn'>Go to MultiCRM Dashboard</a></p>";
    } else {
        echo "<div class='error'><strong>❌ Some tables are missing. Please check the errors above.</strong></div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'><strong>❌ Migration failed:</strong><br>";
    echo htmlspecialchars($e->getMessage()) . "</div>";
}

echo "</div></body></html>";
?>
