<?php
/**
 * Final System Check
 * Verifies all tables and columns are ready
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../config/database.php';

echo "<!DOCTYPE html>
<html lang='tr'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>System Ready Check - MultiCRM v2.0</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; max-width: 1000px; margin: 50px auto; padding: 20px; background: #f5f7fa; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        h1 { color: #1f93ff; margin-top: 0; }
        h2 { color: #334155; border-bottom: 2px solid #e2e8f0; padding-bottom: 10px; }
        .success { background: #dcfce7; color: #166534; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .error { background: #fee2e2; color: #991b1b; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .warning { background: #fef3c7; color: #92400e; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .info { background: #dbeafe; color: #1e40af; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .check-item { display: flex; align-items: center; padding: 8px; margin: 5px 0; }
        .check-item.ok { background: #f0fdf4; border-left: 4px solid #22c55e; }
        .check-item.fail { background: #fef2f2; border-left: 4px solid #ef4444; }
        .btn { background: #1f93ff; color: white; padding: 12px 24px; border: none; border-radius: 6px; cursor: pointer; font-size: 16px; font-weight: 600; text-decoration: none; display: inline-block; margin: 10px 5px; }
        .btn:hover { background: #0066cc; }
        .btn-success { background: #22c55e; }
        .btn-success:hover { background: #16a34a; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; font-size: 13px; }
        th, td { padding: 8px; border: 1px solid #e2e8f0; text-align: left; }
        th { background: #f8fafc; font-weight: 600; }
    </style>
</head>
<body>
    <div class='container'>";

echo "<h1>🎯 MultiCRM v2.0 - System Ready Check</h1>";

$allGood = true;

// Check Tables
echo "<h2>📋 Database Tables</h2>";
$requiredTables = [
    'users' => 'User accounts',
    'customers' => 'Customer records',
    'conversations' => 'Chat conversations',
    'messages' => 'Chat messages',
    'assignment_rules' => 'Assignment rules (NEW)',
    'user_permissions' => 'User permissions (NEW)',
    'activity_logs' => 'Activity logs (NEW)',
    'conversation_transfers' => 'Transfer history (NEW)',
    'agent_availability' => 'Agent status (NEW)',
    'agent_statistics' => 'Performance metrics (NEW)'
];

foreach ($requiredTables as $table => $desc) {
    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM `$table`");
        $count = $stmt->fetchColumn();
        echo "<div class='check-item ok'>✅ <strong>$table</strong> - $desc ($count rows)</div>";
    } catch (PDOException $e) {
        echo "<div class='check-item fail'>❌ <strong>$table</strong> - MISSING</div>";
        $allGood = false;
    }
}

// Check Customers Columns
echo "<h2>👥 Customers Table Columns</h2>";
$requiredCustomerCols = [
    'facebook_id' => 'Facebook user ID',
    'instagram_id' => 'Instagram user ID',
    'preferred_channel' => 'Preferred communication channel',
    'last_seen_at' => 'Last activity timestamp'
];

try {
    $stmt = $pdo->query("DESCRIBE customers");
    $existingCols = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    foreach ($requiredCustomerCols as $col => $desc) {
        if (in_array($col, $existingCols)) {
            echo "<div class='check-item ok'>✅ <strong>$col</strong> - $desc</div>";
        } else {
            echo "<div class='check-item fail'>❌ <strong>$col</strong> - MISSING</div>";
            $allGood = false;
        }
    }
} catch (PDOException $e) {
    echo "<div class='error'>Error checking customers table: " . htmlspecialchars($e->getMessage()) . "</div>";
    $allGood = false;
}

// Check Conversations Columns
echo "<h2>💬 Conversations Table Columns</h2>";
$requiredConvCols = [
    'platform' => 'Message platform (whatsapp/facebook)',
    'assignment_type' => 'How conversation was assigned',
    'assigned_at' => 'Assignment timestamp',
    'first_response_time' => 'First response time in seconds',
    'avg_response_time' => 'Average response time'
];

try {
    $stmt = $pdo->query("DESCRIBE conversations");
    $existingCols = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    foreach ($requiredConvCols as $col => $desc) {
        if (in_array($col, $existingCols)) {
            echo "<div class='check-item ok'>✅ <strong>$col</strong> - $desc</div>";
        } else {
            echo "<div class='check-item fail'>❌ <strong>$col</strong> - MISSING</div>";
            $allGood = false;
        }
    }
} catch (PDOException $e) {
    echo "<div class='error'>Error checking conversations table: " . htmlspecialchars($e->getMessage()) . "</div>";
    $allGood = false;
}

// Check Assignment Rules
echo "<h2>⚙️ Assignment Rules</h2>";
try {
    $stmt = $pdo->query("SELECT * FROM assignment_rules WHERE is_active = 1");
    $rules = $stmt->fetchAll();
    
    if (count($rules) > 0) {
        echo "<div class='success'>✅ Found " . count($rules) . " active assignment rule(s)</div>";
        echo "<table>";
        echo "<tr><th>Name</th><th>Platform</th><th>Type</th><th>Max Conversations</th></tr>";
        foreach ($rules as $rule) {
            echo "<tr>";
            echo "<td>{$rule['name']}</td>";
            echo "<td>{$rule['platform']}</td>";
            echo "<td>{$rule['assignment_type']}</td>";
            echo "<td>{$rule['max_conversations_per_agent']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div class='warning'>⚠️ No active assignment rules found. Creating default rule...</div>";
        try {
            $pdo->exec("INSERT INTO assignment_rules (name, description, platform, assignment_type, priority, is_active) 
                       VALUES ('Default Auto Assignment', 'Automatically assign new conversations using round-robin', 'all', 'round_robin', 1, 1)");
            echo "<div class='success'>✅ Default assignment rule created</div>";
        } catch (PDOException $e) {
            echo "<div class='error'>Error creating rule: " . htmlspecialchars($e->getMessage()) . "</div>";
        }
    }
} catch (PDOException $e) {
    echo "<div class='error'>Error checking assignment rules: " . htmlspecialchars($e->getMessage()) . "</div>";
    $allGood = false;
}

// Check Agent Availability
echo "<h2>👨‍💼 Agent Availability</h2>";
try {
    $stmt = $pdo->query("SELECT u.name, aa.status, aa.current_conversations, aa.max_conversations 
                         FROM users u 
                         LEFT JOIN agent_availability aa ON u.id = aa.user_id 
                         WHERE u.status = 'active'");
    $agents = $stmt->fetchAll();
    
    if (count($agents) > 0) {
        echo "<table>";
        echo "<tr><th>Agent</th><th>Status</th><th>Current</th><th>Max</th></tr>";
        foreach ($agents as $agent) {
            $status = $agent['status'] ?? 'NOT SET';
            $statusClass = $status === 'online' ? 'success' : 'warning';
            echo "<tr>";
            echo "<td>{$agent['name']}</td>";
            echo "<td><span class='$statusClass'>{$status}</span></td>";
            echo "<td>" . ($agent['current_conversations'] ?? 0) . "</td>";
            echo "<td>" . ($agent['max_conversations'] ?? 10) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Create missing availability records
        $missingCount = 0;
        foreach ($agents as $agent) {
            if ($agent['status'] === null) {
                $missingCount++;
            }
        }
        
        if ($missingCount > 0) {
            echo "<div class='warning'>⚠️ $missingCount agent(s) missing availability records. Creating...</div>";
            try {
                $pdo->exec("INSERT INTO agent_availability (user_id, status, max_conversations, auto_accept)
                           SELECT id, 'online', 10, 1 FROM users 
                           WHERE status = 'active' AND id NOT IN (SELECT user_id FROM agent_availability)");
                echo "<div class='success'>✅ Availability records created</div>";
            } catch (PDOException $e) {
                echo "<div class='error'>Error: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
        }
    } else {
        echo "<div class='warning'>⚠️ No active agents found</div>";
    }
} catch (PDOException $e) {
    echo "<div class='error'>Error checking agents: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Final Result
echo "<hr>";
if ($allGood) {
    echo "<div class='success' style='font-size: 18px; text-align: center; padding: 30px;'>";
    echo "<h2 style='margin: 0 0 15px 0;'>🎉 System Ready!</h2>";
    echo "<p>All required tables and columns are in place.</p>";
    echo "<p>MultiCRM v2.0 is ready to use!</p>";
    echo "</div>";
    echo "<div style='text-align: center;'>";
    echo "<a href='../index.php' class='btn btn-success' style='font-size: 18px; padding: 15px 40px;'>🚀 Go to Dashboard</a>";
    echo "</div>";
} else {
    echo "<div class='error' style='font-size: 18px; text-align: center; padding: 30px;'>";
    echo "<h2 style='margin: 0 0 15px 0;'>❌ System Not Ready</h2>";
    echo "<p>Some components are missing. Please check the errors above.</p>";
    echo "</div>";
}

echo "</div></body></html>";
?>
