<?php
/**
 * Fixed Column Updates - Compatible with MySQL 5.7+
 * Checks if column exists before adding
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../config/database.php';

echo "<!DOCTYPE html>
<html lang='tr'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Column Updates - MultiCRM</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; max-width: 900px; margin: 50px auto; padding: 20px; background: #f5f7fa; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        h1 { color: #1f93ff; margin-top: 0; }
        .success { background: #dcfce7; color: #166534; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .error { background: #fee2e2; color: #991b1b; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .warning { background: #fef3c7; color: #92400e; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .info { background: #dbeafe; color: #1e40af; padding: 12px; border-radius: 6px; margin: 10px 0; }
        .btn { background: #1f93ff; color: white; padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block; margin-top: 20px; }
        .btn:hover { background: #0066cc; }
    </style>
</head>
<body>
    <div class='container'>";

echo "<h1>🔧 Column Updates (MySQL 5.7 Compatible)</h1>";

// Helper function to check if column exists
function columnExists($pdo, $table, $column) {
    try {
        $stmt = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
        $stmt->execute([$column]);
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

// Customers table updates
echo "<h2>Updating `customers` table</h2>";

$customerColumns = [
    ['name' => 'facebook_id', 'sql' => "ALTER TABLE `customers` ADD COLUMN `facebook_id` VARCHAR(255) DEFAULT NULL AFTER `email`"],
    ['name' => 'instagram_id', 'sql' => "ALTER TABLE `customers` ADD COLUMN `instagram_id` VARCHAR(255) DEFAULT NULL AFTER `facebook_id`"],
    ['name' => 'preferred_channel', 'sql' => "ALTER TABLE `customers` ADD COLUMN `preferred_channel` ENUM('whatsapp', 'facebook', 'instagram', 'any') DEFAULT 'any' AFTER `instagram_id`"],
    ['name' => 'last_seen_at', 'sql' => "ALTER TABLE `customers` ADD COLUMN `last_seen_at` TIMESTAMP NULL DEFAULT NULL AFTER `preferred_channel`"],
    ['name' => 'language', 'sql' => "ALTER TABLE `customers` ADD COLUMN `language` VARCHAR(10) DEFAULT 'tr' AFTER `last_seen_at`"]
];

foreach ($customerColumns as $col) {
    if (!columnExists($pdo, 'customers', $col['name'])) {
        try {
            $pdo->exec($col['sql']);
            echo "<div class='success'>✅ Added column `{$col['name']}` to customers</div>";
        } catch (PDOException $e) {
            echo "<div class='error'>❌ Error adding `{$col['name']}`: " . htmlspecialchars($e->getMessage()) . "</div>";
        }
    } else {
        echo "<div class='warning'>⚠️ Column `{$col['name']}` already exists in customers</div>";
    }
}

// Add indexes for customers
echo "<h3>Adding indexes to customers</h3>";
try {
    $pdo->exec("ALTER TABLE `customers` ADD INDEX `idx_facebook_id` (`facebook_id`)");
    echo "<div class='success'>✅ Added index on facebook_id</div>";
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate key') !== false) {
        echo "<div class='warning'>⚠️ Index on facebook_id already exists</div>";
    } else {
        echo "<div class='error'>❌ " . htmlspecialchars($e->getMessage()) . "</div>";
    }
}

try {
    $pdo->exec("ALTER TABLE `customers` ADD INDEX `idx_instagram_id` (`instagram_id`)");
    echo "<div class='success'>✅ Added index on instagram_id</div>";
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate key') !== false) {
        echo "<div class='warning'>⚠️ Index on instagram_id already exists</div>";
    }
}

// Conversations table updates
echo "<h2>Updating `conversations` table</h2>";

$conversationColumns = [
    ['name' => 'platform', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `platform` ENUM('whatsapp', 'facebook', 'instagram') DEFAULT 'whatsapp' AFTER `status`"],
    ['name' => 'assignment_type', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `assignment_type` ENUM('manual', 'auto', 'round_robin', 'load_balanced') DEFAULT 'auto' AFTER `platform`"],
    ['name' => 'assigned_at', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `assigned_at` TIMESTAMP NULL DEFAULT NULL AFTER `assignment_type`"],
    ['name' => 'last_agent_message_at', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `last_agent_message_at` TIMESTAMP NULL DEFAULT NULL AFTER `assigned_at`"],
    ['name' => 'last_customer_message_at', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `last_customer_message_at` TIMESTAMP NULL DEFAULT NULL AFTER `last_agent_message_at`"],
    ['name' => 'first_response_time', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `first_response_time` INT DEFAULT NULL COMMENT 'Seconds' AFTER `last_customer_message_at`"],
    ['name' => 'avg_response_time', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `avg_response_time` INT DEFAULT NULL COMMENT 'Seconds' AFTER `first_response_time`"],
    ['name' => 'message_count', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `message_count` INT DEFAULT 0 AFTER `avg_response_time`"],
    ['name' => 'agent_message_count', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `agent_message_count` INT DEFAULT 0 AFTER `message_count`"],
    ['name' => 'customer_message_count', 'sql' => "ALTER TABLE `conversations` ADD COLUMN `customer_message_count` INT DEFAULT 0 AFTER `agent_message_count`"]
];

foreach ($conversationColumns as $col) {
    if (!columnExists($pdo, 'conversations', $col['name'])) {
        try {
            $pdo->exec($col['sql']);
            echo "<div class='success'>✅ Added column `{$col['name']}` to conversations</div>";
        } catch (PDOException $e) {
            echo "<div class='error'>❌ Error adding `{$col['name']}`: " . htmlspecialchars($e->getMessage()) . "</div>";
        }
    } else {
        echo "<div class='warning'>⚠️ Column `{$col['name']}` already exists in conversations</div>";
    }
}

// Add indexes for conversations
echo "<h3>Adding indexes to conversations</h3>";
try {
    $pdo->exec("ALTER TABLE `conversations` ADD INDEX `idx_platform` (`platform`)");
    echo "<div class='success'>✅ Added index on platform</div>";
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate key') !== false) {
        echo "<div class='warning'>⚠️ Index on platform already exists</div>";
    }
}

try {
    $pdo->exec("ALTER TABLE `conversations` ADD INDEX `idx_assignment_type` (`assignment_type`)");
    echo "<div class='success'>✅ Added index on assignment_type</div>";
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate key') !== false) {
        echo "<div class='warning'>⚠️ Index on assignment_type already exists</div>";
    }
}

// Update existing data
echo "<h2>Updating Existing Data</h2>";

try {
    $stmt = $pdo->exec("UPDATE `conversations` SET `platform` = 'whatsapp' WHERE `platform` IS NULL");
    echo "<div class='success'>✅ Set default platform for existing conversations</div>";
} catch (PDOException $e) {
    echo "<div class='warning'>⚠️ " . htmlspecialchars($e->getMessage()) . "</div>";
}

try {
    $stmt = $pdo->exec("UPDATE `conversations` SET `assignment_type` = 'manual' WHERE `assignment_type` IS NULL");
    echo "<div class='success'>✅ Set default assignment_type for existing conversations</div>";
} catch (PDOException $e) {
    echo "<div class='warning'>⚠️ " . htmlspecialchars($e->getMessage()) . "</div>";
}

try {
    $stmt = $pdo->exec("UPDATE `conversations` SET `assigned_at` = `created_at` WHERE `assigned_at` IS NULL AND `status` = 'active'");
    echo "<div class='success'>✅ Set assigned_at for existing conversations</div>";
} catch (PDOException $e) {
    echo "<div class='warning'>⚠️ " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Modify phone column to allow NULL
echo "<h2>Modifying Constraints</h2>";
try {
    $pdo->exec("ALTER TABLE `customers` MODIFY COLUMN `phone` VARCHAR(20) DEFAULT NULL");
    echo "<div class='success'>✅ Modified phone column to allow NULL</div>";
} catch (PDOException $e) {
    echo "<div class='warning'>⚠️ " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Final verification
echo "<h2>🔍 Final Verification</h2>";

$requiredColumns = [
    'customers' => ['facebook_id', 'instagram_id', 'preferred_channel', 'last_seen_at'],
    'conversations' => ['platform', 'assignment_type', 'assigned_at', 'first_response_time']
];

$allColumnsExist = true;
foreach ($requiredColumns as $table => $columns) {
    echo "<h3>Table: $table</h3>";
    foreach ($columns as $column) {
        if (columnExists($pdo, $table, $column)) {
            echo "<div class='success'>✓ Column `$column` exists</div>";
        } else {
            echo "<div class='error'>✗ Column `$column` NOT found</div>";
            $allColumnsExist = false;
        }
    }
}

if ($allColumnsExist) {
    echo "<div class='success'><strong>🎉 All columns updated successfully!</strong></div>";
    echo "<p><a href='../index.php' class='btn'>Go to MultiCRM Dashboard</a></p>";
} else {
    echo "<div class='error'><strong>❌ Some columns are still missing.</strong></div>";
}

echo "</div></body></html>";
?>
