<?php
/**
 * Bilgi Bankası Dosya Yükleme API
 * PDF, TXT, DOCX dosyalarını yükler ve içeriğini extract eder
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit();
}

$user = getUserById($_SESSION['user_id']);
if (!in_array($user['role'], ['admin', 'supervisor'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Forbidden']);
    exit();
}

try {
    $chatbot_id = $_POST['chatbot_id'] ?? 0;
    
    if (!isset($_FILES['file'])) {
        throw new Exception('Dosya yüklenmedi');
    }
    
    $file = $_FILES['file'];
    $file_name = basename($file['name']);
    $file_size = $file['size'];
    $file_tmp = $file['tmp_name'];
    
    // Dosya uzantısını al
    $ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
    
    // Uzantı kontrolü
    if (!in_array($ext, ['pdf', 'txt', 'docx'])) {
        throw new Exception('Desteklenmeyen dosya tipi. Sadece PDF, TXT, DOCX');
    }
    
    // Boyut kontrolü (10MB)
    if ($file_size > 10 * 1024 * 1024) {
        throw new Exception('Dosya çok büyük. Maksimum 10MB');
    }
    
    // Dosya adını güvenli hale getir
    $safe_name = preg_replace('/[^a-zA-Z0-9._-]/', '_', $file_name);
    $unique_name = time() . '_' . $safe_name;
    
    // Yükleme klasörü
    $upload_dir = __DIR__ . '/../../uploads/chatbot_knowledge/';
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $file_path = $upload_dir . $unique_name;
    
    // Dosyayı kaydet
    if (!move_uploaded_file($file_tmp, $file_path)) {
        throw new Exception('Dosya kaydedilemedi');
    }
    
    // İçeriği extract et
    $content = extractFileContent($file_path, $ext);
    
    // Veritabanına kaydet
    $stmt = $pdo->prepare("
        INSERT INTO chatbot_knowledge (chatbot_id, file_name, file_type, file_path, content, file_size)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $chatbot_id,
        $file_name,
        $ext,
        $file_path,
        $content,
        $file_size
    ]);
    
    echo json_encode([
        'success' => true,
        'file_id' => $pdo->lastInsertId()
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

/**
 * Dosya içeriğini extract et
 */
function extractFileContent($file_path, $type) {
    switch ($type) {
        case 'txt':
            return file_get_contents($file_path);
            
        case 'pdf':
            return extractPdfContent($file_path);
            
        case 'docx':
            return extractDocxContent($file_path);
            
        default:
            return '';
    }
}

/**
 * PDF içeriğini extract et
 */
function extractPdfContent($file_path) {
    // Basit PDF text extraction
    // Not: Daha gelişmiş extraction için smalot/pdfparser kütüphanesi kullanılabilir
    
    $content = '';
    
    // pdftotext komutu varsa kullan
    if (shell_exec('which pdftotext')) {
        $output_file = $file_path . '.txt';
        shell_exec("pdftotext '$file_path' '$output_file'");
        
        if (file_exists($output_file)) {
            $content = file_get_contents($output_file);
            unlink($output_file);
        }
    } else {
        // Basit regex ile text extraction (sınırlı)
        $pdf_content = file_get_contents($file_path);
        if (preg_match_all('/\((.*?)\)/s', $pdf_content, $matches)) {
            $content = implode(' ', $matches[1]);
        }
    }
    
    return $content;
}

/**
 * DOCX içeriğini extract et
 */
function extractDocxContent($file_path) {
    $content = '';
    
    // DOCX bir ZIP dosyasıdır
    $zip = new ZipArchive();
    
    if ($zip->open($file_path) === TRUE) {
        // document.xml dosyasını oku
        $xml_content = $zip->getFromName('word/document.xml');
        
        if ($xml_content) {
            // XML'den text'i extract et
            $xml = simplexml_load_string($xml_content);
            
            if ($xml) {
                // Tüm text node'larını al
                $namespaces = $xml->getNamespaces(true);
                $xml->registerXPathNamespace('w', $namespaces['w']);
                
                $texts = $xml->xpath('//w:t');
                foreach ($texts as $text) {
                    $content .= (string)$text . ' ';
                }
            }
        }
        
        $zip->close();
    }
    
    return $content;
}
?>
