<?php
/**
 * Create User API
 * Admin only - creates new user
 */

session_start();
require_once '../config/database.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

// Admin check
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit();
}

$user = getUserById($_SESSION['user_id']);
if ($user['role'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Permission denied']);
    exit();
}

try {
    $name = $_POST['name'] ?? '';
    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';
    $role = $_POST['role'] ?? 'agent';
    $status = $_POST['status'] ?? 'active';
    
    if (empty($name) || empty($email) || empty($password)) {
        throw new Exception('Tüm alanlar gerekli');
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Geçersiz email adresi');
    }
    
    if (strlen($password) < 6) {
        throw new Exception('Şifre en az 6 karakter olmalı');
    }
    
    // Check if email exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        throw new Exception('Bu email adresi zaten kullanılıyor');
    }
    
    // Create user
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $pdo->prepare("
        INSERT INTO users (name, email, password, role, status, created_at) 
        VALUES (?, ?, ?, ?, ?, NOW())
    ");
    
    $stmt->execute([$name, $email, $hashedPassword, $role, $status]);
    $userId = $pdo->lastInsertId();
    
    // Create agent availability record if agent or supervisor
    if (in_array($role, ['agent', 'supervisor'])) {
        $stmt = $pdo->prepare("
            INSERT INTO agent_availability (user_id, status, max_conversations, auto_accept) 
            VALUES (?, 'online', 10, 1)
        ");
        $stmt->execute([$userId]);
    }
    
    echo json_encode([
        'success' => true,
        'user_id' => $userId,
        'message' => 'Kullanıcı başarıyla oluşturuldu'
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
