<?php
/**
 * Müşteri Listesi API - Filtreleme Desteği
 */

session_start();
require_once '../../config/database.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit();
}

try {
    // Filtreler
    $status = $_GET['status'] ?? '';
    $platform = $_GET['platform'] ?? '';
    $agent_id = $_GET['agent_id'] ?? '';
    $date_from = $_GET['date_from'] ?? '';
    $date_to = $_GET['date_to'] ?? '';
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = 20;
    $offset = ($page - 1) * $limit;
    
    // Base query
    $sql = "
        SELECT 
            cust.id,
            cust.name,
            cust.phone,
            cust.email,
            conv.platform,
            conv.customer_status,
            conv.assigned_to,
            u.name as agent_name,
            MAX(m.created_at) as last_message_time,
            COUNT(DISTINCT m.id) as message_count
        FROM customers cust
        INNER JOIN conversations conv ON cust.id = conv.customer_id
        LEFT JOIN users u ON conv.assigned_to = u.id
        LEFT JOIN messages m ON conv.id = m.conversation_id
        WHERE 1=1
    ";
    
    $params = [];
    
    // Status filter
    if ($status) {
        $sql .= " AND conv.customer_status = ?";
        $params[] = $status;
    }
    
    // Platform filter
    if ($platform) {
        $sql .= " AND conv.platform = ?";
        $params[] = $platform;
    }
    
    // Agent filter
    if ($agent_id) {
        $sql .= " AND conv.assigned_to = ?";
        $params[] = $agent_id;
    }
    
    // Date range filter
    if ($date_from) {
        $sql .= " AND DATE(conv.created_at) >= ?";
        $params[] = $date_from;
    }
    
    if ($date_to) {
        $sql .= " AND DATE(conv.created_at) <= ?";
        $params[] = $date_to;
    }
    
    $sql .= " GROUP BY cust.id, cust.name, cust.phone, cust.email, conv.platform, 
              conv.customer_status, conv.assigned_to, u.name";
    $sql .= " ORDER BY last_message_time DESC";
    
    // Count total
    $count_sql = "SELECT COUNT(*) as total FROM (" . $sql . ") as subquery";
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total = $count_stmt->fetch()['total'];
    
    // Get paginated results
    $sql .= " LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $customers = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'customers' => $customers,
        'pagination' => [
            'current_page' => $page,
            'total_pages' => ceil($total / $limit),
            'total_records' => $total,
            'per_page' => $limit
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
