<?php
/**
 * API Ayarlarını Kaydetme Endpoint'i - Simplified Version
 */

// Enable all error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

session_start();
header('Content-Type: application/json');

// Step 1: Check if files can be loaded
$errors = [];

if (!file_exists(__DIR__ . '/../config/database.php')) {
    $errors[] = 'database.php not found';
}

if (!file_exists(__DIR__ . '/../includes/functions.php')) {
    $errors[] = 'functions.php not found';
}

if (!file_exists(__DIR__ . '/../includes/encryption.php')) {
    $errors[] = 'encryption.php not found';
}

if (!empty($errors)) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'File loading errors',
        'errors' => $errors
    ]);
    exit();
}

// Step 2: Try to load files
try {
    require_once __DIR__ . '/../config/database.php';
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to load database.php: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    exit();
}

try {
    require_once __DIR__ . '/../includes/functions.php';
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to load functions.php: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    exit();
}

// Step 3: Check session
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Oturum açmanız gerekiyor.'
    ]);
    exit();
}

// Step 4: Check user
try {
    $user_id = $_SESSION['user_id'];
    $user = getUserById($user_id);
    
    if (!$user) {
        throw new Exception('User not found');
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'User check failed: ' . $e->getMessage()
    ]);
    exit();
}

// Step 5: Check permissions
if (!in_array($user['role'], ['admin', 'supervisor'])) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => 'Bu işlem için yetkiniz yok.'
    ]);
    exit();
}

// Step 6: Check request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Sadece POST istekleri kabul edilir.'
    ]);
    exit();
}

// Step 7: Parse JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Geçersiz JSON verisi: ' . json_last_error_msg()
    ]);
    exit();
}

// Step 8: Validate platform
$platform = $data['platform'] ?? null;

if (!in_array($platform, ['whatsapp', 'facebook', 'instagram', 'google_ai'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Geçersiz platform. whatsapp, facebook, instagram veya google_ai olmalı.',
        'received_platform' => $platform
    ]);
    exit();
}

// Step 9: Check if encryption functions exist
if (!function_exists('saveMultipleSettings')) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'saveMultipleSettings function not found. Encryption.php may not be loaded.'
    ]);
    exit();
}

// Step 10: Save settings
try {
    $pdo->beginTransaction();
    
    if ($platform === 'whatsapp') {
        $settings = [
            'whatsapp_phone_number_id' => [
                'value' => $data['phone_number_id'] ?? '',
                'type' => 'string',
                'description' => 'WhatsApp Phone Number ID',
                'encrypt' => false
            ],
            'whatsapp_business_account_id' => [
                'value' => $data['business_account_id'] ?? '',
                'type' => 'string',
                'description' => 'WhatsApp Business Account ID',
                'encrypt' => false
            ],
            'whatsapp_access_token' => [
                'value' => $data['access_token'] ?? '',
                'type' => 'string',
                'description' => 'WhatsApp Access Token',
                'encrypt' => true
            ],
            'whatsapp_api_version' => [
                'value' => $data['api_version'] ?? 'v18.0',
                'type' => 'string',
                'description' => 'WhatsApp API Version',
                'encrypt' => false
            ],
            'whatsapp_verify_token' => [
                'value' => $data['verify_token'] ?? '',
                'type' => 'string',
                'description' => 'WhatsApp Verify Token',
                'encrypt' => false
            ],
            'whatsapp_webhook_url' => [
                'value' => $data['webhook_url'] ?? '',
                'type' => 'string',
                'description' => 'WhatsApp Webhook URL',
                'encrypt' => false
            ]
        ];
        
        if (!saveMultipleSettings($pdo, $settings)) {
            throw new Exception('WhatsApp ayarları kaydedilemedi.');
        }
        
    } else if ($platform === 'facebook') {
        $settings = [
            'facebook_app_id' => [
                'value' => $data['app_id'] ?? '',
                'type' => 'string',
                'description' => 'Facebook App ID',
                'encrypt' => false
            ],
            'facebook_app_secret' => [
                'value' => $data['app_secret'] ?? '',
                'type' => 'string',
                'description' => 'Facebook App Secret',
                'encrypt' => true
            ],
            'facebook_page_id' => [
                'value' => $data['page_id'] ?? '',
                'type' => 'string',
                'description' => 'Facebook Page ID',
                'encrypt' => false
            ],
            'facebook_page_access_token' => [
                'value' => $data['page_access_token'] ?? '',
                'type' => 'string',
                'description' => 'Facebook Page Access Token',
                'encrypt' => true
            ],
            'facebook_api_version' => [
                'value' => $data['api_version'] ?? 'v18.0',
                'type' => 'string',
                'description' => 'Facebook API Version',
                'encrypt' => false
            ],
            'facebook_verify_token' => [
                'value' => $data['verify_token'] ?? '',
                'type' => 'string',
                'description' => 'Facebook Verify Token',
                'encrypt' => false
            ],
            'facebook_webhook_url' => [
                'value' => $data['webhook_url'] ?? '',
                'type' => 'string',
                'description' => 'Facebook Webhook URL',
                'encrypt' => false
            ]
        ];
        
        if (!saveMultipleSettings($pdo, $settings)) {
            throw new Exception('Facebook ayarları kaydedilemedi.');
        }
    } else if ($platform === 'instagram') {
        $settings = [
            'instagram_business_account_id' => [
                'value' => $data['business_account_id'] ?? '',
                'type' => 'string',
                'description' => 'Instagram Business Account ID',
                'encrypt' => false
            ],
            'instagram_verify_token' => [
                'value' => $data['verify_token'] ?? '',
                'type' => 'string',
                'description' => 'Instagram Verify Token',
                'encrypt' => false
            ],
            'instagram_webhook_url' => [
                'value' => $data['webhook_url'] ?? '',
                'type' => 'string',
                'description' => 'Instagram Webhook URL',
                'encrypt' => false
            ]
        ];
        
        if (!saveMultipleSettings($pdo, $settings)) {
            throw new Exception('Instagram ayarları kaydedilemedi.');
        }
    } else if ($platform === 'google_ai') {
        $settings = [
            'google_ai_api_key' => [
                'value' => $data['api_key'] ?? '',
                'type' => 'string',
                'description' => 'Google Gemini API Key',
                'encrypt' => true
            ],
            'google_ai_model' => [
                'value' => $data['model'] ?? 'gemini-pro',
                'type' => 'string',
                'description' => 'Google Gemini Model',
                'encrypt' => false
            ]
        ];
        
        if (!saveMultipleSettings($pdo, $settings)) {
            throw new Exception('Google AI ayarları kaydedilemedi.');
        }
    }
    
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => ucfirst($platform) . ' ayarları başarıyla kaydedildi.'
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Ayarlar kaydedilirken bir hata oluştu: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ]);
}
?>
