<?php
/**
 * Transfer Conversation API
 * Transfer a conversation from one agent to another
 */

header('Content-Type: application/json');
session_start();

require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../includes/assignment-engine.php';

// Check authentication
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$user_id = $_SESSION['user_id'];
$user = getUserById($user_id);

// Get POST data
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['conversation_id']) || !isset($input['to_agent_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Missing required fields']);
    exit;
}

$conversation_id = (int)$input['conversation_id'];
$to_agent_id = (int)$input['to_agent_id'];
$reason = $input['reason'] ?? null;
$notes = $input['notes'] ?? null;

try {
    // Get conversation
    $conversation = getConversationById($conversation_id);
    
    if (!$conversation) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Conversation not found']);
        exit;
    }
    
    $from_agent_id = $conversation['assigned_to'];
    
    // Check permission
    // Admin and supervisor can transfer any conversation
    // Agent can only transfer their own conversations
    if ($user['role'] === 'agent' && $from_agent_id != $user_id) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'You can only transfer your own conversations']);
        exit;
    }
    
    // Get assignment engine
    $engine = getAssignmentEngine();
    
    // Transfer conversation
    $success = $engine->transferConversation(
        $conversation_id,
        $to_agent_id,
        $from_agent_id,
        $user_id,
        $reason,
        $notes
    );
    
    if ($success) {
        echo json_encode([
            'success' => true,
            'message' => 'Conversation transferred successfully'
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to transfer conversation']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
