<?php
/**
 * WhatsApp Webhook Handler
 * Bu dosya WhatsApp'tan gelen mesajları işler
 */

require_once '../config/database.php';
require_once '../includes/functions.php';

// Log dizinini oluştur
if (!is_dir('../logs')) {
    mkdir('../logs', 0755, true);
}

// Webhook doğrulama (GET isteği)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $mode = $_GET['hub_mode'] ?? '';
    $token = $_GET['hub_verify_token'] ?? '';
    $challenge = $_GET['hub_challenge'] ?? '';
    
    if ($mode === 'subscribe' && verifyWebhook($token, $challenge)) {
        echo $challenge;
        writeLog("Webhook doğrulandı: " . $challenge);
        exit();
    } else {
        http_response_code(403);
        echo "Webhook doğrulama başarısız";
        writeLog("Webhook doğrulama başarısız - Token: " . $token);
        exit();
    }
}

// Webhook verilerini işle (POST isteği)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    writeLog("Webhook verisi alındı: " . $input);
    
    // Webhook verisini logla
    try {
        $stmt = $pdo->prepare("INSERT INTO webhook_logs (webhook_data, processed) VALUES (?, 0)");
        $stmt->execute([$input]);
        $log_id = $pdo->lastInsertId();
    } catch (Exception $e) {
        writeLog("Webhook log hatası: " . $e->getMessage());
    }
    
    if (isset($data['entry'])) {
        foreach ($data['entry'] as $entry) {
            if (isset($entry['changes'])) {
                foreach ($entry['changes'] as $change) {
                    if ($change['field'] === 'messages' && isset($change['value']['messages'])) {
                        processMessages($change['value']['messages'], $change['value']['contacts'] ?? []);
                    }
                }
            }
        }
    }
    
    // Webhook logunu işaretle
    if (isset($log_id)) {
        try {
            $stmt = $pdo->prepare("UPDATE webhook_logs SET processed = 1 WHERE id = ?");
            $stmt->execute([$log_id]);
        } catch (Exception $e) {
            writeLog("Webhook log güncelleme hatası: " . $e->getMessage());
        }
    }
    
    echo "OK";
    exit();
}

/**
 * Gelen mesajları işle
 */
function processMessages($messages, $contacts) {
    global $pdo;
    
    foreach ($messages as $message) {
        try {
            $from = $message['from'];
            $messageId = $message['id'];
            $timestamp = $message['timestamp'];
            $type = $message['type'];
            
            // Müşteri bilgilerini al
            $contact = null;
            foreach ($contacts as $c) {
                if ($c['wa_id'] === $from) {
                    $contact = $c;
                    break;
                }
            }
            
            $customerName = $contact['profile']['name'] ?? 'Bilinmeyen';
            $customerPhone = formatPhoneNumber($from);
            
            // Müşteriyi bul veya oluştur
            $customer = getCustomerByPhone($customerPhone);
            if (!$customer) {
                createCustomer([
                    'name' => $customerName,
                    'phone' => $customerPhone,
                    'email' => null,
                    'facebook_id' => null,
                    'notes' => 'WhatsApp webhook üzerinden eklendi'
                ]);
                $customer = getCustomerByPhone($customerPhone);
                writeLog("Yeni müşteri oluşturuldu: " . $customerName . " (" . $customerPhone . ")");
            }
            
            // Sohbeti bul veya oluştur
            $conversation = getConversationByCustomer($customer['id'], 'whatsapp');
            if (!$conversation) {
                // Yeni sohbet oluştur - assignment engine kullan
                require_once '../includes/assignment-engine.php';
                $engine = getAssignmentEngine();
                
                // Önce sohbeti oluştur (atanmamış)
                $conversationId = createConversation([
                    'customer_id' => $customer['id'],
                    'assigned_to' => 0, // Henüz atanmadı
                    'status' => 'active',
                    'platform' => 'whatsapp'
                ]);
                
                // Otomatik atama yap
                $assigned = $engine->autoAssignConversation($conversationId, 'whatsapp');
                
                if ($assigned) {
                    $conversation = getConversationById($conversationId);
                    writeLog("Yeni sohbet oluşturuldu ve atandı: " . $conversationId);
                } else {
                    // Atama başarısız, sohbet atanmamış olarak kalacak
                    $conversation = getConversationById($conversationId);
                    writeLog("Yeni sohbet oluşturuldu ama atanamadı: " . $conversationId);
                }
            }
            
            // Mesaj içeriğini al
            $content = '';
            if ($type === 'text') {
                $content = $message['text']['body'];
            } elseif ($type === 'image') {
                $content = '[Resim] ' . ($message['image']['caption'] ?? '');
            } elseif ($type === 'document') {
                $content = '[Dosya] ' . ($message['document']['filename'] ?? '');
            } elseif ($type === 'audio') {
                $content = '[Ses Mesajı]';
            } elseif ($type === 'video') {
                $content = '[Video] ' . ($message['video']['caption'] ?? '');
            } elseif ($type === 'location') {
                $content = '[Konum] ' . $message['location']['name'] ?? '';
            } else {
                $content = '[' . ucfirst($type) . ' mesajı]';
            }
            
            // Mesajı kaydet
            saveMessage([
                'conversation_id' => $conversation['id'],
                'sender_id' => null,  // Müşteriden geldiği için user_id yok
                'content' => $content,
                'sender_type' => 'customer',
                'whatsapp_message_id' => $messageId
            ]);
            
            // Müşteri bilgilerini güncelle
            if ($customerName !== 'Bilinmeyen' && $customer['name'] !== $customerName) {
                $stmt = $pdo->prepare("UPDATE customers SET name = ? WHERE id = ?");
                $stmt->execute([$customerName, $customer['id']]);
            }
            
            writeLog("Mesaj işlendi: " . $content . " - Müşteri: " . $customerName);
            
            // Zamanlama kontrolü - AI otomatik aktif/pasif et
            if (function_exists('checkAndUpdateAISchedule')) {
                $result = checkAndUpdateAISchedule($conversation['id']);
                if ($result === true) {
                    writeLog("AI otomatik aktif edildi (zamanlama) - Conversation ID: " . $conversation['id']);
                } else if ($result === false) {
                    writeLog("AI otomatik pasif edildi (zamanlama) - Conversation ID: " . $conversation['id']);
                }
            }
            
            // AI Yanıtı İşle (Google Gemini)
            if (function_exists('processAIResponse')) {
                processAIResponse($conversation['id'], $content, 'whatsapp');
            }
            
        } catch (Exception $e) {
            writeLog("Mesaj işleme hatası: " . $e->getMessage());
        }
    }
}

http_response_code(200);
?>
