<?php
/**
 * Delete Workspace API
 * POST /api/workspaces/delete.php
 */

// Enable error logging
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

session_start();
require_once '../../config/database.php';
require_once '../../includes/workspace.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

try {
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Unauthorized - Giriş yapmanız gerekiyor']);
        exit;
    }

    $user = getUserById($_SESSION['user_id']);

    if (!$user) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Kullanıcı bulunamadı']);
        exit;
    }

    // Only admin can delete workspaces
    if ($user['role'] !== 'admin') {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Bu işlem için admin yetkisi gerekiyor']);
        exit;
    }

    // Get POST data
    $input = file_get_contents('php://input');
    error_log("Delete workspace input: " . $input);
    
    $data = json_decode($input, true);

    if (!isset($data['workspace_id'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'workspace_id parametresi gerekli']);
        exit;
    }

    $workspace_id = (int)$data['workspace_id'];
    
    error_log("Attempting to delete workspace: " . $workspace_id);

    // Check if workspace exists
    $stmt = $pdo->prepare("SELECT * FROM agencies WHERE id = ?");
    $stmt->execute([$workspace_id]);
    $workspace = $stmt->fetch();

    if (!$workspace) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Workspace bulunamadı (ID: ' . $workspace_id . ')']);
        exit;
    }

    // Prevent deleting the last workspace
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM agencies WHERE status = 'active'");
    $stmt->execute();
    $result = $stmt->fetch();

    if ($result['count'] <= 1) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Son workspace silinemez. En az bir workspace olmalı.']);
        exit;
    }

    $pdo->beginTransaction();
    
    // Soft delete - just mark as inactive
    $stmt = $pdo->prepare("UPDATE agencies SET status = 'inactive', updated_at = NOW() WHERE id = ?");
    $stmt->execute([$workspace_id]);
    
    error_log("Workspace marked as inactive: " . $workspace_id);
    
    // If current workspace is being deleted, switch to another one
    if (isset($_SESSION['workspace_id']) && $_SESSION['workspace_id'] == $workspace_id) {
        error_log("Current workspace is being deleted, switching to another");
        
        $defaultWorkspace = getDefaultWorkspace($_SESSION['user_id']);
        if ($defaultWorkspace && $defaultWorkspace['id'] != $workspace_id) {
            $_SESSION['workspace_id'] = $defaultWorkspace['id'];
            $_SESSION['workspace_name'] = $defaultWorkspace['name'];
            error_log("Switched to default workspace: " . $defaultWorkspace['id']);
        } else {
            // Get any other active workspace
            $stmt = $pdo->prepare("
                SELECT a.* FROM agencies a
                INNER JOIN user_agencies ua ON a.id = ua.agency_id
                WHERE ua.user_id = ? AND a.status = 'active' AND a.id != ?
                LIMIT 1
            ");
            $stmt->execute([$_SESSION['user_id'], $workspace_id]);
            $altWorkspace = $stmt->fetch();
            
            if ($altWorkspace) {
                $_SESSION['workspace_id'] = $altWorkspace['id'];
                $_SESSION['workspace_name'] = $altWorkspace['name'];
                error_log("Switched to alternative workspace: " . $altWorkspace['id']);
            } else {
                error_log("WARNING: No alternative workspace found");
            }
        }
    }
    
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Workspace başarıyla silindi'
    ]);
    
} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("Error deleting workspace: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => 'Workspace silinemedi: ' . $e->getMessage()
    ]);
}
?>
