<?php
/**
 * Update Workspace API
 * POST /api/workspaces/update.php
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/workspace.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

$user = getUserById($_SESSION['user_id']);

// Only admin can update workspaces
if ($user['role'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Bu işlem için yetkiniz yok']);
    exit;
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

if (!isset($data['workspace_id']) || !isset($data['name'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'workspace_id ve name gerekli']);
    exit;
}

$workspace_id = (int)$data['workspace_id'];
$name = sanitizeInput($data['name']);
$domain = isset($data['domain']) ? sanitizeInput($data['domain']) : null;

if (empty($name)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Workspace adı boş olamaz']);
    exit;
}

// Check if workspace exists
$stmt = $pdo->prepare("SELECT * FROM agencies WHERE id = ?");
$stmt->execute([$workspace_id]);
$workspace = $stmt->fetch();

if (!$workspace) {
    http_response_code(404);
    echo json_encode(['success' => false, 'error' => 'Workspace bulunamadı']);
    exit;
}

try {
    $stmt = $pdo->prepare("
        UPDATE agencies 
        SET name = ?, domain = ?, updated_at = NOW() 
        WHERE id = ?
    ");
    $stmt->execute([$name, $domain, $workspace_id]);
    
    // Update session if current workspace
    if ($_SESSION['workspace_id'] == $workspace_id) {
        $_SESSION['workspace_name'] = $name;
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Workspace başarıyla güncellendi',
        'workspace' => [
            'id' => $workspace_id,
            'name' => $name,
            'domain' => $domain
        ]
    ]);
    
} catch (Exception $e) {
    error_log("Error updating workspace: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Workspace güncellenemedi']);
}
?>

