/**
 * Customer Info Page JavaScript
 */

let currentPage = 1;
let currentFilters = {};

// Sayfa yüklendiğinde müşterileri yükle
document.addEventListener('DOMContentLoaded', function () {
    loadCustomers();
});

// Müşterileri yükle
async function loadCustomers(page = 1) {
    currentPage = page;
    const tbody = document.getElementById('customer-table-body');
    tbody.innerHTML = '<tr><td colspan="7" class="text-center"><i class="fas fa-spinner fa-spin"></i> Yükleniyor...</td></tr>';

    try {
        // Build query string
        const params = new URLSearchParams({
            page: page,
            ...currentFilters
        });

        const response = await fetch(`api/customers/list.php?${params}`);
        const data = await response.json();

        if (data.success) {
            renderCustomers(data.customers);
            renderPagination(data.pagination);
        } else {
            tbody.innerHTML = `<tr><td colspan="7" class="text-center text-danger">Hata: ${data.error}</td></tr>`;
        }
    } catch (error) {
        console.error('Error loading customers:', error);
        tbody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Bir hata oluştu</td></tr>';
        tbody.innerHTML = '<tr><td colspan="6" class="text-center text-danger">Bir hata oluştu</td></tr>';
    }
}

// Müşterileri render et
function renderCustomers(customers) {
    const tbody = document.getElementById('customer-table-body');

    if (customers.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center">Müşteri bulunamadı</td></tr>';
        return;
    }

    tbody.innerHTML = customers.map(customer => `
        <tr>
            <td>
                <strong>${escapeHtml(customer.name || 'İsimsiz')}</strong><br>
                <small class="text-muted">${escapeHtml(customer.phone || '-')}</small>
            </td>
            <td>
                ${getPlatformBadge(customer.platform)}
            </td>
            <td>
                ${getStatusBadge(customer.customer_status)}
            </td>
            <td>
                ${customer.agent_name ? `<i class="fas fa-user"></i> ${escapeHtml(customer.agent_name)}` : '<span class="text-muted">Atanmamış</span>'}
            </td>
            <td>
                ${customer.last_message_time ? formatDate(customer.last_message_time) : '-'}
            </td>
            <td>
                <span class="badge bg-primary">${customer.message_count || 0}</span>
            </td>
        </tr>
    `).join('');
}

// Platform badge
function getPlatformBadge(platform) {
    const platforms = {
        whatsapp: { icon: 'fab fa-whatsapp', class: 'platform-whatsapp', text: 'WhatsApp' },
        facebook: { icon: 'fab fa-facebook', class: 'platform-facebook', text: 'Facebook' },
        instagram: { icon: 'fab fa-instagram', class: 'platform-instagram', text: 'Instagram' }
    };

    const p = platforms[platform] || { icon: 'fas fa-question', class: '', text: platform || '-' };
    return `<span class="platform-badge ${p.class}"><i class="${p.icon}"></i> ${p.text}</span>`;
}

// Status badge
function getStatusBadge(status) {
    const statuses = {
        potential: { class: 'status-potential', text: 'Potansiyel' },
        info_received: { class: 'status-info_received', text: 'Bilgi Alındı' },
        positive: { class: 'status-positive', text: 'Pozitif' },
        negative: { class: 'status-negative', text: 'Negatif' },
        pending: { class: 'status-pending', text: 'Beklemede' }
    };

    const s = statuses[status] || { class: '', text: '-' };
    return status ? `<span class="status-badge ${s.class}">${s.text}</span>` : '-';
}

// Pagination render
function renderPagination(pagination) {
    const container = document.getElementById('pagination');

    if (pagination.total_pages <= 1) {
        container.innerHTML = '';
        return;
    }

    let html = '';

    // Previous button
    if (pagination.current_page > 1) {
        html += `<button onclick="loadCustomers(${pagination.current_page - 1})"><i class="fas fa-chevron-left"></i></button>`;
    }

    // Page numbers
    for (let i = 1; i <= pagination.total_pages; i++) {
        if (i === pagination.current_page) {
            html += `<button class="active">${i}</button>`;
        } else if (i === 1 || i === pagination.total_pages || Math.abs(i - pagination.current_page) <= 2) {
            html += `<button onclick="loadCustomers(${i})">${i}</button>`;
        } else if (Math.abs(i - pagination.current_page) === 3) {
            html += '<button disabled>...</button>';
        }
    }

    // Next button
    if (pagination.current_page < pagination.total_pages) {
        html += `<button onclick="loadCustomers(${pagination.current_page + 1})"><i class="fas fa-chevron-right"></i></button>`;
    }

    container.innerHTML = html;
}

// Filtreleri uygula
function applyFilters() {
    currentFilters = {
        status: document.getElementById('status-filter').value,
        platform: document.getElementById('platform-filter').value,
        agent_id: document.getElementById('agent-filter').value,
        date_from: document.getElementById('date-from').value,
        date_to: document.getElementById('date-to').value
    };

    // Boş filtreleri kaldır
    Object.keys(currentFilters).forEach(key => {
        if (!currentFilters[key]) delete currentFilters[key];
    });

    loadCustomers(1);
}

// Filtreleri temizle
function clearFilters() {
    document.getElementById('status-filter').value = '';
    document.getElementById('platform-filter').value = '';
    document.getElementById('agent-filter').value = '';
    document.getElementById('date-from').value = '';
    document.getElementById('date-to').value = '';

    currentFilters = {};
    loadCustomers(1);
}

// Müşteri detaylarını görüntüle
function viewCustomer(customerId) {
    // Modal'ı aç ve müşteri detaylarını yükle
    const modal = document.getElementById('customer-modal');
    modal.style.display = 'block';

    // Detayları yükle
    fetch(`api/customers/get.php?id=${customerId}`)
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                document.getElementById('modal-customer-name').textContent = data.customer.name;
                document.getElementById('modal-customer-details').innerHTML = `
                    <p><strong>Telefon:</strong> ${data.customer.phone || '-'}</p>
                    <p><strong>Email:</strong> ${data.customer.email || '-'}</p>
                    <p><strong>Platform:</strong> ${getPlatformBadge(data.customer.platform)}</p>
                    <p><strong>Durum:</strong> ${getStatusBadge(data.customer.customer_status)}</p>
                    <p><strong>Atanan Ajan:</strong> ${data.customer.agent_name || 'Atanmamış'}</p>
                `;
            }
        })
        .catch(error => console.error('Error:', error));
}

// Modal'ı kapat
function closeCustomerModal() {
    document.getElementById('customer-modal').style.display = 'none';
}

// Export customers
function exportCustomers() {
    const params = new URLSearchParams(currentFilters);
    window.location.href = `api/customers/export.php?${params}`;
}

// Utility functions
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diff = now - date;
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));

    if (days === 0) {
        return 'Bugün ' + date.toLocaleTimeString('tr-TR', { hour: '2-digit', minute: '2-digit' });
    } else if (days === 1) {
        return 'Dün ' + date.toLocaleTimeString('tr-TR', { hour: '2-digit', minute: '2-digit' });
    } else if (days < 7) {
        return days + ' gün önce';
    } else {
        return date.toLocaleDateString('tr-TR');
    }
}

// Modal dışına tıklandığında kapat
window.onclick = function (event) {
    const modal = document.getElementById('customer-modal');
    if (event.target === modal) {
        modal.style.display = 'none';
    }
}
