/**
 * MultiCRM - Gerçek Zamanlı Özellikler
 * WebSocket alternatifi olarak polling ve Server-Sent Events kullanır
 */

class RealtimeManager {
    constructor() {
        this.pollingInterval = null;
        this.eventSource = null;
        this.isOnline = navigator.onLine;
        this.lastCheckTime = new Date().toISOString();
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.startPolling();
        this.setupOfflineHandling();
    }

    setupEventListeners() {
        // Online/Offline durumu
        window.addEventListener('online', () => {
            this.isOnline = true;
            this.showNotification('Bağlantı yeniden kuruldu', 'success');
            this.startPolling();
        });

        window.addEventListener('offline', () => {
            this.isOnline = false;
            this.showNotification('Bağlantı kesildi', 'warning');
            this.stopPolling();
        });

        // Sayfa görünürlüğü
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                this.pausePolling();
            } else {
                this.resumePolling();
            }
        });
    }

    startPolling() {
        if (this.pollingInterval) return;
        
        this.pollingInterval = setInterval(() => {
            this.checkForUpdates();
        }, 2000); // 2 saniyede bir kontrol
    }

    stopPolling() {
        if (this.pollingInterval) {
            clearInterval(this.pollingInterval);
            this.pollingInterval = null;
        }
    }

    pausePolling() {
        this.stopPolling();
    }

    resumePolling() {
        if (this.isOnline) {
            this.startPolling();
        }
    }

    async checkForUpdates() {
        if (!this.isOnline) return;

        try {
            const response = await fetch(`api/poll-messages.php?last_check=${this.lastCheckTime}`, {
                method: 'GET',
                headers: {
                    'Cache-Control': 'no-cache',
                    'Pragma': 'no-cache'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }

            const data = await response.json();
            
            if (data.success) {
                this.lastCheckTime = data.timestamp;
                this.handleUpdates(data);
            }
        } catch (error) {
            console.error('Güncelleme kontrolü hatası:', error);
            this.handleConnectionError();
        }
    }

    handleUpdates(data) {
        // Yeni mesajlar
        if (data.notifications && data.notifications.length > 0) {
            this.showNewMessages(data.notifications);
        }

        // İstatistikler
        if (data.stats) {
            this.updateStats(data.stats);
        }

        // Sohbet listesini güncelle
        if (window.multicrm && typeof window.multicrm.loadConversations === 'function') {
            window.multicrm.loadConversations();
        }
    }

    showNewMessages(notifications) {
        notifications.forEach(notification => {
            // Browser bildirimi
            this.showBrowserNotification(notification);
            
            // Sayfa içi bildirim
            this.showInPageNotification(notification);
            
            // Ses bildirimi
            this.playNotificationSound();
        });
    }

    showBrowserNotification(notification) {
        if (Notification.permission === 'granted') {
            const notification = new Notification(notification.customer_name, {
                body: notification.message,
                icon: '/multicrm/assets/images/whatsapp-icon.png',
                badge: '/multicrm/assets/images/badge-icon.png',
                tag: `conversation_${notification.conversation_id}`,
                requireInteraction: true
            });

            notification.onclick = () => {
                window.focus();
                // Sohbeti aç
                if (window.multicrm && typeof window.multicrm.selectConversation === 'function') {
                    window.multicrm.selectConversation(notification.conversation_id);
                }
                notification.close();
            };
        }
    }

    showInPageNotification(notification) {
        const notificationElement = document.createElement('div');
        notificationElement.className = 'realtime-notification';
        notificationElement.innerHTML = `
            <div class="notification-content">
                <i class="fab fa-whatsapp"></i>
                <div>
                    <strong>${notification.customer_name}</strong>
                    <p>${notification.message}</p>
                </div>
                <button class="btn-close" onclick="this.parentElement.parentElement.remove()">×</button>
            </div>
        `;

        document.body.appendChild(notificationElement);

        // 5 saniye sonra otomatik kaldır
        setTimeout(() => {
            if (notificationElement.parentElement) {
                notificationElement.remove();
            }
        }, 5000);
    }

    playNotificationSound() {
        // Basit ses bildirimi
        const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBSuBzvLZiTYIG2m98OScTgwOUarm7blbGgU7k9n1unEiBC13yO/eizEIHWq+8+OWT');
        audio.volume = 0.3;
        audio.play().catch(e => console.log('Ses çalınamadı:', e));
    }

    updateStats(stats) {
        // Bildirim sayacını güncelle
        const badge = document.querySelector('.notification-badge');
        if (badge) {
            const total = stats.new_messages + stats.new_conversations;
            badge.textContent = total > 0 ? total : '';
            badge.style.display = total > 0 ? 'block' : 'none';
        }
    }

    handleConnectionError() {
        this.showNotification('Bağlantı hatası, yeniden deneniyor...', 'warning');
    }

    setupOfflineHandling() {
        // Offline durumda mesajları localStorage'da sakla
        window.addEventListener('beforeunload', () => {
            if (!this.isOnline) {
                localStorage.setItem('multicrm_offline_messages', JSON.stringify({
                    timestamp: new Date().toISOString(),
                    messages: []
                }));
            }
        });
    }

    showNotification(message, type = 'info') {
        if (window.multicrm && typeof window.multicrm.showNotification === 'function') {
            window.multicrm.showNotification(message, type);
        } else {
            console.log(`[${type.toUpperCase()}] ${message}`);
        }
    }
}

// CSS stilleri ekle
const style = document.createElement('style');
style.textContent = `
.realtime-notification {
    position: fixed;
    top: 20px;
    right: 20px;
    background: white;
    border: 1px solid #ddd;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    z-index: 10000;
    max-width: 300px;
    animation: slideIn 0.3s ease;
}

@keyframes slideIn {
    from { transform: translateX(100%); opacity: 0; }
    to { transform: translateX(0); opacity: 1; }
}

.notification-content {
    display: flex;
    align-items: center;
    padding: 12px;
    gap: 10px;
}

.notification-content i {
    color: #25D366;
    font-size: 1.2rem;
}

.notification-content div {
    flex: 1;
}

.notification-content strong {
    display: block;
    font-size: 0.9rem;
    color: #333;
}

.notification-content p {
    margin: 0;
    font-size: 0.8rem;
    color: #666;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.btn-close {
    background: none;
    border: none;
    font-size: 1.2rem;
    color: #999;
    cursor: pointer;
    padding: 0;
    width: 20px;
    height: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.notification-badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #ff4444;
    color: white;
    border-radius: 50%;
    width: 18px;
    height: 18px;
    font-size: 0.7rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
}
`;
document.head.appendChild(style);

// Gerçek zamanlı yöneticiyi başlat
document.addEventListener('DOMContentLoaded', () => {
    window.realtimeManager = new RealtimeManager();
});
