/**
 * Workspace Switcher JavaScript
 * Handles workspace switching functionality
 */

document.addEventListener('DOMContentLoaded', function () {
    const currentWorkspace = document.querySelector('.current-workspace');
    const workspaceDropdown = document.querySelector('.workspace-dropdown');
    const workspaceItems = document.querySelectorAll('.workspace-item');

    // Toggle dropdown
    if (currentWorkspace && workspaceDropdown) {
        currentWorkspace.addEventListener('click', function (e) {
            e.stopPropagation();
            workspaceDropdown.style.display = workspaceDropdown.style.display === 'none' ? 'block' : 'none';

            // Toggle icon rotation
            const icon = currentWorkspace.querySelector('.workspace-dropdown-icon i');
            if (icon) {
                icon.style.transform = workspaceDropdown.style.display === 'none' ? 'rotate(0deg)' : 'rotate(180deg)';
            }
        });
    }

    // Close dropdown when clicking outside
    document.addEventListener('click', function (e) {
        if (workspaceDropdown && !e.target.closest('.workspace-selector')) {
            workspaceDropdown.style.display = 'none';
            const icon = currentWorkspace?.querySelector('.workspace-dropdown-icon i');
            if (icon) {
                icon.style.transform = 'rotate(0deg)';
            }
        }
    });

    // Handle workspace switching
    workspaceItems.forEach(item => {
        item.addEventListener('click', async function () {
            const workspaceId = this.dataset.workspaceId;

            if (!workspaceId) return;

            // Show loading state
            this.style.opacity = '0.5';
            this.style.pointerEvents = 'none';

            try {
                const response = await fetch('api/workspaces/switch.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ workspace_id: parseInt(workspaceId) })
                });

                const data = await response.json();

                if (data.success) {
                    // Reload page to reflect new workspace
                    window.location.reload();
                } else {
                    alert('Workspace değiştirilemedi: ' + (data.error || 'Bilinmeyen hata'));
                    this.style.opacity = '1';
                    this.style.pointerEvents = 'auto';
                }
            } catch (error) {
                console.error('Workspace switching error:', error);
                alert('Bir hata oluştu. Lütfen tekrar deneyin.');
                this.style.opacity = '1';
                this.style.pointerEvents = 'auto';
            }
        });
    });
});
