<?php
/**
 * Workspace Data Isolation Check & Fix
 * Bu script workspace_id kolonlarının varlığını kontrol eder ve eksikleri tamamlar
 */

session_start();
require_once 'config/database.php';

header('Content-Type: text/html; charset=utf-8');

echo "<!DOCTYPE html><html><head><meta charset='utf-8'><title>Workspace Veri İzolasyonu Kontrolü</title>";
echo "<style>
body { font-family: Arial; padding: 20px; background: #f5f5f5; }
.box { background: white; padding: 20px; margin: 10px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
.success { color: #28a745; }
.warning { color: #ffc107; }
.error { color: #dc3545; }
.info { color: #17a2b8; }
h2 { color: #333; border-bottom: 2px solid #007bff; padding-bottom: 10px; }
pre { background: #f8f9fa; padding: 10px; border-left: 3px solid #007bff; overflow-x: auto; }
</style></head><body>";

echo "<h1>🔍 Workspace Veri İzolasyonu Kontrolü</h1>";

try {
    // 1. Check customers table
    echo "<div class='box'>";
    echo "<h2>1️⃣ Customers Tablosu</h2>";
    
    $stmt = $pdo->query("SHOW COLUMNS FROM customers LIKE 'workspace_id'");
    $hasWorkspaceId = $stmt->fetch();
    
    if ($hasWorkspaceId) {
        echo "<p class='success'>✅ workspace_id kolonu mevcut</p>";
        
        // Count customers per workspace
        $stmt = $pdo->query("
            SELECT c.workspace_id, a.name as workspace_name, COUNT(*) as count 
            FROM customers c
            LEFT JOIN agencies a ON c.workspace_id = a.id
            GROUP BY c.workspace_id
        ");
        $counts = $stmt->fetchAll();
        
        echo "<h3>Workspace Bazlı Müşteri Dağılımı:</h3><pre>";
        foreach ($counts as $row) {
            echo "Workspace ID: {$row['workspace_id']} ({$row['workspace_name']}) - {$row['count']} müşteri\n";
        }
        echo "</pre>";
    } else {
        echo "<p class='error'>❌ workspace_id kolonu eksik!</p>";
        echo "<p class='info'>🔧 Düzeltme uygulanıyor...</p>";
        
        $pdo->exec("
            ALTER TABLE customers 
            ADD COLUMN workspace_id INT NOT NULL DEFAULT 1 AFTER id,
            ADD INDEX idx_workspace_id (workspace_id),
            ADD FOREIGN KEY fk_customer_workspace (workspace_id) REFERENCES agencies(id) ON DELETE RESTRICT
        ");
        
        echo "<p class='success'>✅ workspace_id kolonu eklendi!</p>";
    }
    echo "</div>";
    
    // 2. Check conversations table
    echo "<div class='box'>";
    echo "<h2>2️⃣ Conversations Tablosu</h2>";
    
    $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
    $hasWorkspaceId = $stmt->fetch();
    
    if ($hasWorkspaceId) {
        echo "<p class='success'>✅ workspace_id kolonu mevcut</p>";
        
        // Count conversations per workspace
        $stmt = $pdo->query("
            SELECT c.workspace_id, a.name as workspace_name, COUNT(*) as count 
            FROM conversations c
            LEFT JOIN agencies a ON c.workspace_id = a.id
            GROUP BY c.workspace_id
        ");
        $counts = $stmt->fetchAll();
        
        echo "<h3>Workspace Bazlı Sohbet Dağılımı:</h3><pre>";
        foreach ($counts as $row) {
            echo "Workspace ID: {$row['workspace_id']} ({$row['workspace_name']}) - {$row['count']} sohbet\n";
        }
        echo "</pre>";
    } else {
        echo "<p class='error'>❌ workspace_id kolonu eksik!</p>";
        echo "<p class='info'>🔧 Düzeltme uygulanıyor...</p>";
        
        $pdo->exec("
            ALTER TABLE conversations 
            ADD COLUMN workspace_id INT NOT NULL DEFAULT 1 AFTER id,
            ADD INDEX idx_workspace_id (workspace_id),
            ADD FOREIGN KEY fk_conversation_workspace (workspace_id) REFERENCES agencies(id) ON DELETE RESTRICT
        ");
        
        echo "<p class='success'>✅ workspace_id kolonu eklendi!</p>";
    }
    echo "</div>";
    
    // 3. Check if filters are applied in functions
    echo "<div class='box'>";
    echo "<h2>3️⃣ Fonksiyon Kontrolleri</h2>";
    
    echo "<h3>Kontrol Edilmesi Gereken Fonksiyonlar:</h3>";
    echo "<ul>";
    echo "<li><code>getActiveConversations()</code> - Workspace filtresi eklenecek</li>";
    echo "<li><code>getAllConversations()</code> - Workspace filtresi eklenecek</li>";
    echo "<li><code>getCustomerByPhone()</code> - Workspace filtresi eklenecek</li>";
    echo "<li><code>customers.php</code> - SQL sorgusu workspace filtresi eklenecek</li>";
    echo "</ul>";
    
    echo "<p class='warning'>⚠️ Bu fonksiyonlar workspace bazlı filtreleme için güncellenecek</p>";
    echo "</div>";
    
    // 4. Workspace count
    echo "<div class='box'>";
    echo "<h2>4️⃣ Mevcut Workspace'ler</h2>";
    
    $stmt = $pdo->query("SELECT id, name, domain, status FROM agencies ORDER BY id");
    $workspaces = $stmt->fetchAll();
    
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr style='background: #007bff; color: white;'><th style='padding:10px;'>ID</th><th style='padding:10px;'>İsim</th><th style='padding:10px;'>Domain</th><th style='padding:10px;'>Durum</th></tr>";
    foreach ($workspaces as $ws) {
        $statusColor = $ws['status'] === 'active' ? 'green' : 'red';
        echo "<tr>";
        echo "<td style='padding:10px;'>{$ws['id']}</td>";
        echo "<td style='padding:10px;'><strong>{$ws['name']}</strong></td>";
        echo "<td style='padding:10px;'>{$ws['domain']}</td>";
        echo "<td style='padding:10px; color:{$statusColor};'>{$ws['status']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    
    // 5. Recommendations
    echo "<div class='box'>";
    echo "<h2>5️⃣ Öneriler</h2>";
    echo "<h3>🎯 Workspace Bazlı Veri İzolasyonu için Yapılacaklar:</h3>";
    echo "<ol>";
    echo "<li><strong>Fonksiyonları Güncelle:</strong> <code>includes/functions.php</code> dosyasındaki tüm SELECT sorgularına workspace_id filtresi ekle</li>";
    echo "<li><strong>Müşteriler Sayfası:</strong> <code>customers.php</code> - Sadece aktif workspace'in müşterilerini göster</li>";
    echo "<li><strong>Sohbetler Sayfası:</strong> <code>index.php</code> - Sadece aktif workspace'in sohbetlerini göster</li>";
    echo "<li><strong>API Endpoint'leri:</strong> Tüm API'lerde workspace kontrolü ekle</li>";
    echo "<li><strong>Webhook'lar:</strong> Gelen mesajları doğru workspace'e yönlendir</li>";
    echo "</ol>";
    echo "</div>";
    
    echo "<div class='box' style='background: #d4edda; border-left: 4px solid #28a745;'>";
    echo "<h2>✅ Sonraki Adım</h2>";
    echo "<p><strong>Şimdi kod güncellemelerini yapabilirim!</strong></p>";
    echo "<p>Workspace bazlı veri izolasyonunu tüm sayfalara ve fonksiyonlara uygulayacağım.</p>";
    echo "</div>";
    
} catch (PDOException $e) {
    echo "<div class='box' style='background: #f8d7da; border-left: 4px solid #dc3545;'>";
    echo "<h2>❌ Hata</h2>";
    echo "<p><strong>Hata Mesajı:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "</body></html>";
?>

