<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'includes/workspace.php';

// Eğer kullanıcı giriş yapmamışsa login sayfasına yönlendir
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user = getUserById($user_id);
$workspace_id = $_SESSION['workspace_id'] ?? 1;

// Check if workspace_id column exists
$hasWorkspaceColumn = false;
try {
    $stmt = $pdo->query("SHOW COLUMNS FROM customers LIKE 'workspace_id'");
    $hasWorkspaceColumn = ($stmt->rowCount() > 0);
} catch (Exception $e) {
    // Column doesn't exist
}

// Müşterileri getir (workspace bazlı)
$sql = "
    SELECT c.*, 
           (SELECT COUNT(*) FROM conversations WHERE customer_id = c.id AND status = 'active') as active_conversations,
           (SELECT created_at FROM conversations WHERE customer_id = c.id ORDER BY created_at DESC LIMIT 1) as last_conversation
    FROM customers c 
    WHERE c.status = 'active'
";

if ($hasWorkspaceColumn) {
    $sql .= " AND c.workspace_id = ?";
}

$sql .= " ORDER BY c.created_at DESC";

$stmt = $pdo->prepare($sql);
if ($hasWorkspaceColumn) {
    $stmt->execute([$workspace_id]);
} else {
    $stmt->execute();
}
$customers = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Müşteriler - MultiCRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/sidebar.php'; ?>
    
    <div class="main-content">
        <div class="container-fluid" style="padding: 20px;">
            <!-- Page Header -->
            <div class="page-header-mini" style="margin-bottom: 30px;">
                <div style="display: flex; justify-content: space-between; align-items: center; gap: 20px;">
                    <h2 style="margin: 0; display: flex; align-items: center; gap: 12px;">
                        <i class="fas fa-users"></i>
                        <span>Müşteriler (<?php echo count($customers); ?>)</span>
                    </h2>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCustomerModal">
                        <i class="fas fa-user-plus"></i> Yeni Müşteri
                    </button>
                </div>
            </div>

            <!-- Müşteriler Tablosu -->
            <div class="card">
                <div class="card-body" style="padding: 0;">
                    <?php if (empty($customers)): ?>
                        <div style="text-align: center; padding: 40px; color: #999;">
                            <i class="fas fa-users" style="font-size: 48px; margin-bottom: 20px; display: block; opacity: 0.3;"></i>
                            <p>Henüz müşteri yok</p>
                        </div>
                    <?php else: ?>
                        <table class="table table-hover" style="margin-bottom: 0;">
                            <thead style="background: #f8f9fa; border-bottom: 2px solid #dee2e6;">
                                <tr>
                                    <th style="padding: 15px; font-weight: 600;">Adı</th>
                                    <th style="padding: 15px; font-weight: 600;">Telefon</th>
                                    <th style="padding: 15px; font-weight: 600;">E-posta</th>
                                    <th style="padding: 15px; font-weight: 600;">Sohbetler</th>
                                    <th style="padding: 15px; font-weight: 600; text-align: right;">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($customers as $customer): ?>
                                    <tr style="border-bottom: 1px solid #dee2e6;">
                                        <td style="padding: 15px; vertical-align: middle;">
                                            <strong><?php echo htmlspecialchars($customer['name']); ?></strong>
                                        </td>
                                        <td style="padding: 15px; vertical-align: middle;">
                                            <?php echo htmlspecialchars($customer['phone']); ?>
                                        </td>
                                        <td style="padding: 15px; vertical-align: middle;">
                                            <?php echo htmlspecialchars($customer['email']); ?>
                                        </td>
                                        <td style="padding: 15px; vertical-align: middle;">
                                            <span class="badge bg-primary"><?php echo $customer['active_conversations']; ?></span>
                                        </td>
                                        <td style="padding: 15px; vertical-align: middle; text-align: right;">
                                            <button class="btn btn-sm btn-outline-primary" 
                                                    onclick="editCustomer(<?php echo $customer['id']; ?>)"
                                                    title="Düzenle">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-sm btn-outline-danger"
                                                    onclick="deleteCustomer(<?php echo $customer['id']; ?>)"
                                                    title="Sil">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Customer Modal -->
    <div class="modal fade" id="addCustomerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Yeni Müşteri</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="add-customer-form">
                        <div class="mb-3">
                            <label for="customer-name" class="form-label">Ad Soyad</label>
                            <input type="text" class="form-control" id="customer-name" required>
                        </div>
                        <div class="mb-3">
                            <label for="customer-phone" class="form-label">Telefon</label>
                            <input type="tel" class="form-control" id="customer-phone" placeholder="+90 5XX XXX XX XX" required>
                        </div>
                        <div class="mb-3">
                            <label for="customer-email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="customer-email">
                        </div>
                        <div class="mb-3">
                            <label for="customer-notes" class="form-label">Notlar</label>
                            <textarea class="form-control" id="customer-notes" rows="3"></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-primary" onclick="saveCustomer()">Kaydet</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Customer Modal -->
    <div class="modal fade" id="editCustomerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Müşteri Düzenle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="edit-customer-form">
                        <input type="hidden" id="edit-customer-id">
                        <div class="mb-3">
                            <label for="edit-customer-name" class="form-label">Ad Soyad</label>
                            <input type="text" class="form-control" id="edit-customer-name" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit-customer-phone" class="form-label">Telefon</label>
                            <input type="tel" class="form-control" id="edit-customer-phone" placeholder="+90 5XX XXX XX XX" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit-customer-email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="edit-customer-email">
                        </div>
                        <div class="mb-3">
                            <label for="edit-customer-notes" class="form-label">Notlar</label>
                            <textarea class="form-control" id="edit-customer-notes" rows="3"></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-primary" onclick="updateCustomer()">Güncelle</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Customer Modal -->
    <div class="modal fade" id="deleteCustomerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Müşteri Sil</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="text-center">
                        <div class="delete-icon mb-3">
                            <i class="fas fa-exclamation-triangle"></i>
                        </div>
                        <h4>Bu müşteriyi silmek istediğinizden emin misiniz?</h4>
                        <p class="text-muted">Bu işlem geri alınamaz. Müşteri ve tüm sohbet geçmişi silinecektir.</p>
                        <div class="alert alert-warning">
                            <i class="fas fa-info-circle"></i>
                            <strong>Dikkat:</strong> Bu müşteriye ait tüm sohbetler ve mesajlar da silinecektir.
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-danger" onclick="confirmDeleteCustomer()">
                        <i class="fas fa-trash"></i> Müşteriyi Sil
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function startConversation(customerId) {
            // Sohbet başlatma işlemi
            window.location.href = `index.php?customer_id=${customerId}`;
        }
        
        function editCustomer(customerId) {
            // Müşteri bilgilerini al
            fetch(`api/get-customer.php?id=${customerId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Modal'ı doldur
                    document.getElementById('edit-customer-id').value = data.customer.id;
                    document.getElementById('edit-customer-name').value = data.customer.name;
                    document.getElementById('edit-customer-phone').value = data.customer.phone;
                    document.getElementById('edit-customer-email').value = data.customer.email || '';
                    document.getElementById('edit-customer-notes').value = data.customer.notes || '';
                    
                    // Modal'ı göster
                    const editModal = new bootstrap.Modal(document.getElementById('editCustomerModal'));
                    editModal.show();
                } else {
                    alert('Hata: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu');
            });
        }
        
        function saveCustomer() {
            const name = document.getElementById('customer-name').value;
            const phone = document.getElementById('customer-phone').value;
            const email = document.getElementById('customer-email').value;
            const notes = document.getElementById('customer-notes').value;
            
            if (!name || !phone) {
                alert('Ad ve telefon alanları zorunludur');
                return;
            }
            
            fetch('api/save-customer.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    name: name,
                    phone: phone,
                    email: email,
                    notes: notes
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Hata: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu');
            });
        }
        
        function updateCustomer() {
            const id = document.getElementById('edit-customer-id').value;
            const name = document.getElementById('edit-customer-name').value;
            const phone = document.getElementById('edit-customer-phone').value;
            const email = document.getElementById('edit-customer-email').value;
            const notes = document.getElementById('edit-customer-notes').value;
            
            if (!name || !phone) {
                alert('Ad ve telefon alanları zorunludur');
                return;
            }
            
            fetch('api/update-customer.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    id: id,
                    name: name,
                    phone: phone,
                    email: email,
                    notes: notes
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Hata: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu');
            });
        }
        
        let customerToDelete = null;
        
        function deleteCustomer(customerId) {
            customerToDelete = customerId;
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteCustomerModal'));
            deleteModal.show();
        }
        
        function confirmDeleteCustomer() {
            if (!customerToDelete) {
                return;
            }
            
            fetch('api/delete-customer.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    id: customerToDelete
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Modal'ı kapat
                    const deleteModal = bootstrap.Modal.getInstance(document.getElementById('deleteCustomerModal'));
                    deleteModal.hide();
                    
                    // Sayfayı yenile
                    location.reload();
                } else {
                    alert('Hata: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu');
            });
        }
        
    </script>
</body>
</html>
