<?php
/**
 * Facebook Webhook Diagnostics
 * Check webhook configuration and recent logs
 */

require_once '../config/database.php';
require_once '../includes/functions.php';

session_start();
if (!isset($_SESSION['user_id'])) {
    die('Unauthorized');
}

$user = getUserById($_SESSION['user_id']);
if (!in_array($user['role'], ['admin', 'supervisor'])) {
    die('Permission denied');
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Facebook Webhook Diagnostics</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { padding: 20px; font-family: monospace; }
        .success { color: #22c55e; }
        .error { color: #ef4444; }
        .warning { color: #f59e0b; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 6px; }
        .section { margin: 20px 0; padding: 20px; border: 1px solid #e2e8f0; border-radius: 8px; }
    </style>
</head>
<body>
    <h1>🔍 Facebook Webhook Diagnostics</h1>

    <div class="section">
        <h3>1. Webhook URL</h3>
        <pre><?php echo 'https://profaj.co/multicrm/api/webhook-facebook.php'; ?></pre>
        <p>Bu URL'yi Facebook App Settings → Webhooks bölümünde kullanın</p>
    </div>

    <div class="section">
        <h3>2. Facebook Configuration</h3>
        <?php if (isset($facebook_config)): ?>
            <p class="success">✅ Facebook config loaded</p>
            <pre><?php 
                $safe_config = $facebook_config;
                $safe_config['page_access_token'] = substr($safe_config['page_access_token'], 0, 20) . '...';
                $safe_config['app_secret'] = '***';
                echo json_encode($safe_config, JSON_PRETTY_PRINT); 
            ?></pre>
        <?php else: ?>
            <p class="error">❌ Facebook config NOT loaded</p>
        <?php endif; ?>
    </div>

    <div class="section">
        <h3>3. Database Settings</h3>
        <?php
        try {
            $stmt = $pdo->query("SELECT * FROM settings WHERE `key` LIKE 'facebook_%'");
            $settings = $stmt->fetchAll();
            
            if (count($settings) > 0) {
                echo '<p class="success">✅ Facebook settings in database: ' . count($settings) . '</p>';
                echo '<pre>';
                foreach ($settings as $setting) {
                    $value = $setting['value'];
                    if (strlen($value) > 50) {
                        $value = substr($value, 0, 20) . '... (' . strlen($value) . ' chars)';
                    }
                    echo $setting['key'] . ' = ' . $value . "\n";
                }
                echo '</pre>';
            } else {
                echo '<p class="warning">⚠️ No Facebook settings in database</p>';
            }
        } catch (Exception $e) {
            echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
        }
        ?>
    </div>

    <div class="section">
        <h3>4. Recent Webhook Logs</h3>
        <?php
        $logFile = '../logs/facebook_webhook_' . date('Y-m-d') . '.log';
        if (file_exists($logFile)) {
            $logs = file_get_contents($logFile);
            $lines = explode("\n", $logs);
            $recent = array_slice(array_reverse($lines), 0, 50);
            echo '<p class="success">✅ Log file exists: ' . basename($logFile) . '</p>';
            echo '<pre style="max-height: 400px; overflow-y: auto;">';
            echo htmlspecialchars(implode("\n", array_reverse($recent)));
            echo '</pre>';
        } else {
            echo '<p class="warning">⚠️ No log file for today</p>';
            echo '<p>Expected: ' . $logFile . '</p>';
            
            // Check for yesterday's log
            $yesterdayLog = '../logs/facebook_webhook_' . date('Y-m-d', strtotime('-1 day')) . '.log';
            if (file_exists($yesterdayLog)) {
                echo '<p class="info">Found yesterday\'s log:</p>';
                $logs = file_get_contents($yesterdayLog);
                $lines = explode("\n", $logs);
                $recent = array_slice(array_reverse($lines), 0, 20);
                echo '<pre style="max-height: 200px; overflow-y: auto;">';
                echo htmlspecialchars(implode("\n", array_reverse($recent)));
                echo '</pre>';
            }
        }
        ?>
    </div>

    <div class="section">
        <h3>5. Recent Facebook Customers</h3>
        <?php
        try {
            $stmt = $pdo->query("SELECT * FROM customers WHERE facebook_id IS NOT NULL ORDER BY created_at DESC LIMIT 10");
            $customers = $stmt->fetchAll();
            
            if (count($customers) > 0) {
                echo '<p class="success">✅ Found ' . count($customers) . ' Facebook customers</p>';
                echo '<table class="table table-sm">';
                echo '<tr><th>ID</th><th>Name</th><th>Facebook ID</th><th>Created</th></tr>';
                foreach ($customers as $c) {
                    echo '<tr>';
                    echo '<td>' . $c['id'] . '</td>';
                    echo '<td>' . htmlspecialchars($c['name']) . '</td>';
                    echo '<td>' . $c['facebook_id'] . '</td>';
                    echo '<td>' . $c['created_at'] . '</td>';
                    echo '</tr>';
                }
                echo '</table>';
            } else {
                echo '<p class="warning">⚠️ No Facebook customers found</p>';
            }
        } catch (Exception $e) {
            echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
        }
        ?>
    </div>

    <div class="section">
        <h3>6. Test Webhook Manually</h3>
        <button class="btn btn-primary" onclick="testWebhook()">Send Test Request</button>
        <div id="test-result" style="margin-top: 10px;"></div>
    </div>

    <script>
        async function testWebhook() {
            const result = document.getElementById('test-result');
            result.innerHTML = '<p>Sending test request...</p>';
            
            try {
                const response = await fetch('../api/webhook-facebook.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        entry: [{
                            id: 'test',
                            messaging: [{
                                sender: {id: '123456789'},
                                recipient: {id: '987654321'},
                                timestamp: Date.now(),
                                message: {
                                    mid: 'test_mid',
                                    text: 'Test message from diagnostics'
                                }
                            }]
                        }]
                    })
                });
                
                const text = await response.text();
                result.innerHTML = '<pre>' + text + '</pre>';
            } catch (error) {
                result.innerHTML = '<p class="error">Error: ' + error.message + '</p>';
            }
        }
    </script>

    <p style="margin-top: 40px;">
        <a href="../index.php" class="btn btn-secondary">← Back to Dashboard</a>
    </p>
</body>
</html>
