<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

try {
    require_once 'config/database.php';
    require_once 'includes/functions.php';
    
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit();
    }

    $user_id = $_SESSION['user_id'];
    $user = getUserById($user_id);
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    $workspace_id = $_SESSION['workspace_id'] ?? 1;

    // Only admin can manage inboxes
    if ($user['role'] !== 'admin') {
        header('Location: index.php');
        exit();
    }
} catch (Exception $e) {
    die("Error loading page: " . $e->getMessage() . "<br><br>Please check your database connection and make sure all required files exist.");
}

$message = '';
$error = '';

// Check if inboxes table exists
$hasInboxesTable = false;
try {
    $stmt = $pdo->query("SHOW TABLES LIKE 'inboxes'");
    $hasInboxesTable = ($stmt->rowCount() > 0);
} catch (Exception $e) {
    $error = 'Database error: ' . $e->getMessage();
}

// Get inboxes for current workspace
$inboxes = [];
if ($hasInboxesTable) {
    try {
        $stmt = $pdo->prepare("
            SELECT i.*, 
                   COUNT(DISTINCT c.id) as conversation_count,
                   COUNT(DISTINCT im.user_id) as member_count
            FROM inboxes i
            LEFT JOIN conversations c ON i.id = c.inbox_id AND c.status = 'active'
            LEFT JOIN inbox_members im ON i.id = im.inbox_id
            WHERE i.workspace_id = ? AND i.is_active = 1
            GROUP BY i.id
            ORDER BY i.created_at DESC
        ");
        $stmt->execute([$workspace_id]);
        $inboxes = $stmt->fetchAll();
    } catch (Exception $e) {
        $error = 'Error fetching inboxes: ' . $e->getMessage();
    }
}

// Get current workspace info
$stmt = $pdo->prepare("SELECT * FROM agencies WHERE id = ?");
$stmt->execute([$workspace_id]);
$workspace = $stmt->fetch();

// Facebook App credentials (admin should set these)
$stmt = $pdo->prepare("SELECT value FROM settings WHERE `key` = 'facebook_app_id' LIMIT 1");
$stmt->execute();
$result = $stmt->fetch();
$fb_app_id = $result ? $result['value'] : '';

$stmt = $pdo->prepare("SELECT value FROM settings WHERE `key` = 'facebook_app_secret' LIMIT 1");
$stmt->execute();
$result = $stmt->fetch();
$fb_app_secret = $result ? $result['value'] : '';

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kanallar (Inboxes) - MultiCRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <style>
        .inbox-card {
            transition: all 0.3s ease;
            border-left: 4px solid transparent;
        }
        .inbox-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        .inbox-card.whatsapp { border-left-color: #25D366; }
        .inbox-card.facebook { border-left-color: #1877F2; }
        .inbox-card.instagram { border-left-color: #E4405F; }
        
        .connect-btn {
            padding: 15px 30px;
            font-size: 16px;
            font-weight: 600;
            border-radius: 10px;
            transition: all 0.3s;
        }
        .connect-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        .btn-facebook {
            background: #1877F2;
            border: none;
            color: white;
        }
        .btn-instagram {
            background: linear-gradient(45deg, #F58529, #DD2A7B, #8134AF);
            border: none;
            color: white;
        }
        .btn-whatsapp {
            background: #25D366;
            border: none;
            color: white;
        }
        .channel-icon {
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            font-size: 28px;
            margin-right: 15px;
        }
        .channel-icon.whatsapp { background: #e8f5e9; color: #25D366; }
        .channel-icon.facebook { background: #e3f2fd; color: #1877F2; }
        .channel-icon.instagram { background: #fce4ec; color: #E4405F; }
        
        .setup-warning {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 15px;
            margin-bottom: 30px;
        }
    </style>
</head>
<body>
    <?php include 'includes/sidebar.php'; ?>
    
    <div class="main-content">
        <div class="container-fluid" style="padding: 20px;">
            
            <!-- Page Header -->
            <div class="page-header-mini" style="margin-bottom: 30px;">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <h2 style="margin: 0; display: flex; align-items: center; gap: 12px;">
                            <i class="fas fa-inbox"></i>
                            <span>Kanallar (Inboxes)</span>
                        </h2>
                        <p class="text-muted" style="margin: 5px 0 0 0; font-size: 14px;">
                            <?php echo htmlspecialchars($workspace['name']); ?> için sosyal medya hesaplarını bağlayın
                        </p>
                    </div>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <?php if (!$hasInboxesTable): ?>
                <!-- Migration Required Warning -->
                <div class="setup-warning">
                    <h3><i class="fas fa-database"></i> Database Kurulumu Gerekli</h3>
                    <p style="margin: 15px 0;">
                        Multi-tenant inbox sistemi için database migration'ını çalıştırmanız gerekiyor.
                    </p>
                    <a href="install/setup-multi-tenant.php" class="btn btn-light btn-lg">
                        <i class="fas fa-play-circle"></i> Migration'ı Çalıştır
                    </a>
                </div>
            <?php elseif (empty($fb_app_id) || empty($fb_app_secret)): ?>
                <!-- Facebook App Setup Required -->
                <div class="setup-warning">
                    <h3><i class="fab fa-facebook"></i> Facebook App Kurulumu Gerekli</h3>
                    <p style="margin: 15px 0;">
                        Facebook ve Instagram bağlantısı için önce Meta (Facebook) Developer hesabından bir App oluşturmanız gerekiyor.
                    </p>
                    <ol style="margin: 15px 0; padding-left: 20px;">
                        <li>Meta Developer Console'a gidin: <a href="https://developers.facebook.com/apps" target="_blank" style="color: white; text-decoration: underline;">developers.facebook.com/apps</a></li>
                        <li>Yeni bir App oluşturun (Business type)</li>
                        <li>WhatsApp, Messenger ve Instagram products ekleyin</li>
                        <li>App ID ve App Secret'i kopyalayın</li>
                        <li>OAuth Redirect URI ekleyin: <code style="background: rgba(255,255,255,0.2); padding: 5px 10px; border-radius: 5px;"><?php echo (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . '/multicrm/oauth/facebook/callback.php'; ?></code></li>
                    </ol>
                    <button class="btn btn-light btn-lg" data-bs-toggle="modal" data-bs-target="#appConfigModal">
                        <i class="fas fa-cog"></i> App ID & Secret Ayarla
                    </button>
                </div>
            <?php endif; ?>

            <!-- Connected Inboxes -->
            <?php if (!empty($inboxes)): ?>
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="fas fa-plug"></i> Bağlı Kanallar</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <?php foreach ($inboxes as $inbox): ?>
                                <div class="col-md-6 mb-3">
                                    <div class="inbox-card card <?php echo $inbox['channel_type']; ?>">
                                        <div class="card-body">
                                            <div class="d-flex align-items-center">
                                                <div class="channel-icon <?php echo $inbox['channel_type']; ?>">
                                                    <i class="fab fa-<?php echo $inbox['channel_type']; ?>"></i>
                                                </div>
                                                <div class="flex-grow-1">
                                                    <h5 class="mb-1"><?php echo htmlspecialchars($inbox['name']); ?></h5>
                                                    <p class="text-muted mb-2" style="font-size: 13px;">
                                                        <i class="fas fa-hashtag"></i> <?php echo htmlspecialchars($inbox['identifier']); ?>
                                                    </p>
                                                    <div style="display: flex; gap: 15px; font-size: 13px;">
                                                        <span><i class="fas fa-comments"></i> <?php echo $inbox['conversation_count']; ?> konuşma</span>
                                                        <span><i class="fas fa-users"></i> <?php echo $inbox['member_count']; ?> üye</span>
                                                    </div>
                                                </div>
                                                <div>
                                                    <button class="btn btn-sm btn-outline-danger" onclick="disconnectInbox(<?php echo $inbox['id']; ?>)">
                                                        <i class="fas fa-unlink"></i> Bağlantıyı Kes
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Connect New Channel -->
            <?php if ($hasInboxesTable && !empty($fb_app_id)): ?>
                <div class="card">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="fas fa-plus-circle"></i> Yeni Kanal Bağla</h5>
                    </div>
                    <div class="card-body">
                        <div class="row g-4">
                            <!-- Facebook Page -->
                            <div class="col-md-4">
                                <div class="text-center p-4">
                                    <div class="channel-icon facebook mx-auto mb-3">
                                        <i class="fab fa-facebook-messenger"></i>
                                    </div>
                                    <h5>Facebook Messenger</h5>
                                    <p class="text-muted">Facebook sayfanıza gelen mesajları yönetin</p>
                                    <button class="connect-btn btn-facebook w-100" onclick="connectFacebook()">
                                        <i class="fab fa-facebook"></i> Facebook ile Bağlan
                                    </button>
                                </div>
                            </div>

                            <!-- Instagram -->
                            <div class="col-md-4">
                                <div class="text-center p-4">
                                    <div class="channel-icon instagram mx-auto mb-3">
                                        <i class="fab fa-instagram"></i>
                                    </div>
                                    <h5>Instagram Direct</h5>
                                    <p class="text-muted">Instagram hesabınıza gelen DM'leri yönetin</p>
                                    <button class="connect-btn btn-instagram w-100" onclick="connectInstagram()">
                                        <i class="fab fa-instagram"></i> Instagram ile Bağlan
                                    </button>
                                </div>
                            </div>

                            <!-- WhatsApp -->
                            <div class="col-md-4">
                                <div class="text-center p-4">
                                    <div class="channel-icon whatsapp mx-auto mb-3">
                                        <i class="fab fa-whatsapp"></i>
                                    </div>
                                    <h5>WhatsApp Business</h5>
                                    <p class="text-muted">WhatsApp Business API hesabınızı bağlayın</p>
                                    <button class="connect-btn btn-whatsapp w-100" data-bs-toggle="modal" data-bs-target="#whatsappModal">
                                        <i class="fab fa-whatsapp"></i> WhatsApp Ayarla
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- App Config Modal -->
    <div class="modal fade" id="appConfigModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Facebook App Ayarları</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="appConfigForm">
                        <div class="mb-3">
                            <label class="form-label">Facebook App ID</label>
                            <input type="text" class="form-control" name="app_id" value="<?php echo htmlspecialchars($fb_app_id); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Facebook App Secret</label>
                            <input type="text" class="form-control" name="app_secret" value="<?php echo htmlspecialchars($fb_app_secret); ?>" required>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-primary" onclick="saveAppConfig()">Kaydet</button>
                </div>
            </div>
        </div>
    </div>

    <!-- WhatsApp Setup Modal -->
    <div class="modal fade" id="whatsappModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title"><i class="fab fa-whatsapp"></i> WhatsApp Business API</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="whatsappForm">
                        <div class="mb-3">
                            <label class="form-label">Inbox Adı</label>
                            <input type="text" class="form-control" name="inbox_name" placeholder="Örn: Müşteri Destek Hattı" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Phone Number ID</label>
                            <input type="text" class="form-control" name="phone_number_id" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Business Account ID</label>
                            <input type="text" class="form-control" name="business_account_id" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Access Token</label>
                            <textarea class="form-control" name="access_token" rows="3" required></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Verify Token (Webhook için)</label>
                            <input type="text" class="form-control" name="verify_token" value="<?php echo bin2hex(random_bytes(16)); ?>" required>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-success" onclick="saveWhatsApp()">
                        <i class="fas fa-save"></i> Kaydet ve Bağlan
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const FB_APP_ID = '<?php echo $fb_app_id; ?>';
        const REDIRECT_URI = '<?php echo (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . '/multicrm/oauth/facebook/callback.php'; ?>';

        function connectFacebook() {
            const scope = 'pages_messaging,pages_manage_metadata,pages_read_engagement,pages_manage_posts,instagram_basic,instagram_manage_messages';
            const authUrl = `https://www.facebook.com/v18.0/dialog/oauth?client_id=${FB_APP_ID}&redirect_uri=${encodeURIComponent(REDIRECT_URI)}&scope=${scope}&state=facebook_${Date.now()}`;
            
            window.location.href = authUrl;
        }

        function connectInstagram() {
            const scope = 'instagram_basic,instagram_manage_messages,instagram_manage_comments,pages_manage_metadata,pages_read_engagement';
            const authUrl = `https://www.facebook.com/v18.0/dialog/oauth?client_id=${FB_APP_ID}&redirect_uri=${encodeURIComponent(REDIRECT_URI)}&scope=${scope}&state=instagram_${Date.now()}`;
            
            window.location.href = authUrl;
        }

        async function saveAppConfig() {
            const formData = new FormData(document.getElementById('appConfigForm'));
            
            try {
                const response = await fetch('api/inboxes/save-app-config.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('App ayarları kaydedildi!');
                    location.reload();
                } else {
                    alert('Hata: ' + result.error);
                }
            } catch (error) {
                alert('Bir hata oluştu: ' + error.message);
            }
        }

        async function saveWhatsApp() {
            const formData = new FormData(document.getElementById('whatsappForm'));
            
            try {
                const response = await fetch('api/inboxes/create-whatsapp.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('WhatsApp bağlantısı başarıyla oluşturuldu!');
                    location.reload();
                } else {
                    alert('Hata: ' + result.error);
                }
            } catch (error) {
                alert('Bir hata oluştu: ' + error.message);
            }
        }

        async function disconnectInbox(inboxId) {
            if (!confirm('Bu inbox bağlantısını kesmek istediğinizden emin misiniz?')) {
                return;
            }
            
            try {
                const response = await fetch('api/inboxes/disconnect.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({inbox_id: inboxId})
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('Inbox bağlantısı kesildi');
                    location.reload();
                } else {
                    alert('Hata: ' + result.error);
                }
            } catch (error) {
                alert('Bir hata oluştu: ' + error.message);
            }
        }
    </script>
</body>
</html>

