<?php
/**
 * Encryption Helper Functions
 * Güvenli şifreleme ve çözme işlemleri için yardımcı fonksiyonlar
 */

// Şifreleme anahtarı - Güvenli bir yerde saklanmalı (environment variable önerilir)
define('ENCRYPTION_KEY', getenv('ENCRYPTION_KEY') ?: 'multicrm_secret_key_2024_profaj_secure');
define('ENCRYPTION_METHOD', 'AES-256-CBC');

/**
 * Veriyi şifreler
 * 
 * @param string $data Şifrelenecek veri
 * @return string|false Şifrelenmiş veri veya hata durumunda false
 */
function encryptData($data) {
    if (empty($data)) {
        return $data;
    }
    
    try {
        $key = hash('sha256', ENCRYPTION_KEY);
        $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length(ENCRYPTION_METHOD));
        
        $encrypted = openssl_encrypt($data, ENCRYPTION_METHOD, $key, 0, $iv);
        
        if ($encrypted === false) {
            error_log('Encryption failed: ' . openssl_error_string());
            return false;
        }
        
        // IV'yi şifrelenmiş verinin başına ekle
        return base64_encode($iv . $encrypted);
    } catch (Exception $e) {
        error_log('Encryption error: ' . $e->getMessage());
        return false;
    }
}

/**
 * Şifrelenmiş veriyi çözer
 * 
 * @param string $encryptedData Şifrelenmiş veri
 * @return string|false Çözülmüş veri veya hata durumunda false
 */
function decryptData($encryptedData) {
    if (empty($encryptedData)) {
        return $encryptedData;
    }
    
    try {
        $key = hash('sha256', ENCRYPTION_KEY);
        $data = base64_decode($encryptedData);
        
        if ($data === false) {
            return false;
        }
        
        $ivLength = openssl_cipher_iv_length(ENCRYPTION_METHOD);
        $iv = substr($data, 0, $ivLength);
        $encrypted = substr($data, $ivLength);
        
        $decrypted = openssl_decrypt($encrypted, ENCRYPTION_METHOD, $key, 0, $iv);
        
        if ($decrypted === false) {
            error_log('Decryption failed: ' . openssl_error_string());
            return false;
        }
        
        return $decrypted;
    } catch (Exception $e) {
        error_log('Decryption error: ' . $e->getMessage());
        return false;
    }
}

/**
 * Ayarı veritabanına kaydeder (şifreleyerek)
 * 
 * @param PDO $pdo Veritabanı bağlantısı
 * @param string $key Ayar anahtarı
 * @param mixed $value Ayar değeri
 * @param string $type Veri tipi (string, number, boolean, json)
 * @param string $description Açıklama
 * @param bool $encrypt Şifrelensin mi?
 * @return bool Başarılı ise true
 */
function saveSetting($pdo, $key, $value, $type = 'string', $description = null, $encrypt = false) {
    try {
        // Şifreleme gerekiyorsa
        if ($encrypt && !empty($value)) {
            $originalValue = $value;
            $value = encryptData($value);
            if ($value === false) {
                error_log("Encryption failed for key: $key, original value length: " . strlen($originalValue));
                throw new Exception("Şifreleme başarısız: $key");
            }
        }
        
        // JSON tipindeyse encode et
        if ($type === 'json' && is_array($value)) {
            $value = json_encode($value);
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO settings (`key`, `value`, `type`, `description`) 
            VALUES (:key, :value, :type, :description)
            ON DUPLICATE KEY UPDATE 
                `value` = VALUES(`value`),
                `type` = VALUES(`type`),
                `description` = VALUES(`description`),
                `updated_at` = CURRENT_TIMESTAMP
        ");
        
        $result = $stmt->execute([
            'key' => $key,
            'value' => $value,
            'type' => $type,
            'description' => $description
        ]);
        
        if (!$result) {
            error_log("Database execute failed for key: $key");
        }
        
        return $result;
    } catch (PDOException $e) {
        error_log("Save setting PDO error for key $key: " . $e->getMessage());
        throw $e;
    } catch (Exception $e) {
        error_log("Save setting error for key $key: " . $e->getMessage());
        throw $e;
    }
}

/**
 * Ayarı veritabanından okur (şifreli ise çözer)
 * 
 * @param PDO $pdo Veritabanı bağlantısı
 * @param string $key Ayar anahtarı
 * @param mixed $default Varsayılan değer
 * @param bool $decrypt Şifreli mi?
 * @return mixed Ayar değeri
 */
function getSetting($pdo, $key, $default = null, $decrypt = false) {
    try {
        $stmt = $pdo->prepare("SELECT `value`, `type` FROM settings WHERE `key` = :key LIMIT 1");
        $stmt->execute(['key' => $key]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$result) {
            return $default;
        }
        
        $value = $result['value'];
        
        // Şifreli ise çöz
        if ($decrypt && !empty($value)) {
            $decrypted = decryptData($value);
            if ($decrypted !== false) {
                $value = $decrypted;
            }
        }
        
        // JSON tipindeyse decode et
        if ($result['type'] === 'json') {
            $decoded = json_decode($value, true);
            return $decoded !== null ? $decoded : $default;
        }
        
        // Boolean tipindeyse dönüştür
        if ($result['type'] === 'boolean') {
            return filter_var($value, FILTER_VALIDATE_BOOLEAN);
        }
        
        // Number tipindeyse dönüştür
        if ($result['type'] === 'number') {
            return is_numeric($value) ? (strpos($value, '.') !== false ? floatval($value) : intval($value)) : $default;
        }
        
        return $value;
    } catch (PDOException $e) {
        error_log('Get setting error: ' . $e->getMessage());
        return $default;
    }
}

/**
 * Birden fazla ayarı toplu olarak kaydeder
 * 
 * @param PDO $pdo Veritabanı bağlantısı
 * @param array $settings Ayarlar dizisi [key => [value, type, description, encrypt]]
 * @return bool Tüm ayarlar başarılı ise true
 */
function saveMultipleSettings($pdo, $settings) {
    foreach ($settings as $key => $config) {
        $value = $config['value'] ?? null;
        $type = $config['type'] ?? 'string';
        $description = $config['description'] ?? null;
        $encrypt = $config['encrypt'] ?? false;
        
        try {
            if (!saveSetting($pdo, $key, $value, $type, $description, $encrypt)) {
                throw new Exception("Failed to save setting: $key");
            }
        } catch (Exception $e) {
            error_log("saveMultipleSettings error for key $key: " . $e->getMessage());
            throw new Exception("Ayar kaydedilemedi ($key): " . $e->getMessage());
        }
    }
    
    return true;
}

/**
 * Birden fazla ayarı toplu olarak okur
 * 
 * @param PDO $pdo Veritabanı bağlantısı
 * @param array $keys Ayar anahtarları dizisi [key => [default, decrypt]]
 * @return array Ayarlar dizisi
 */
function getMultipleSettings($pdo, $keys) {
    $settings = [];
    
    foreach ($keys as $key => $config) {
        $default = $config['default'] ?? null;
        $decrypt = $config['decrypt'] ?? false;
        
        $settings[$key] = getSetting($pdo, $key, $default, $decrypt);
    }
    
    return $settings;
}
?>
