<?php
// Kullanıcı işlemleri
function getUserById($user_id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    return $stmt->fetch();
}

function authenticateUser($email, $password) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND status = 'active'");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if ($user && password_verify($password, $user['password'])) {
        return $user;
    }
    return false;
}

function createUser($name, $email, $password, $role = 'agent') {
    global $pdo;
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $pdo->prepare("INSERT INTO users (name, email, password, role, status, created_at) VALUES (?, ?, ?, ?, 'active', NOW())");
    return $stmt->execute([$name, $email, $hashedPassword, $role]);
}

// Müşteri işlemleri
function getCustomerByPhone($phone, $workspace_id = null) {
    global $pdo;
    
    // Auto-detect workspace from session if not provided
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    if ($workspace_id) {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE phone = ? AND workspace_id = ?");
        $stmt->execute([$phone, $workspace_id]);
    } else {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE phone = ?");
        $stmt->execute([$phone]);
    }
    
    return $stmt->fetch();
}

function createCustomer($data) {
    global $pdo;
    
    // Auto-detect workspace from session if not in data
    $workspace_id = $data['workspace_id'] ?? null;
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    // Default to workspace 1 if still not set (backward compatibility)
    if (!$workspace_id) {
        $workspace_id = 1;
    }
    
    // Eğer array değilse eski format için dönüştür
    if (!is_array($data)) {
        $data = [
            'name' => $data,
            'phone' => func_get_arg(1),
            'email' => func_get_arg(2) ?? null,
            'facebook_id' => null,
            'notes' => null
        ];
    }
    
    // Facebook müşterileri için telefon numarası yoksa Facebook ID kullan
    $phone = $data['phone'] ?? null;
    if (empty($phone) && !empty($data['facebook_id'])) {
        $phone = 'fb_' . $data['facebook_id'];
    }
    
    $stmt = $pdo->prepare("INSERT INTO customers (workspace_id, name, phone, email, facebook_id, notes, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    $stmt->execute([
        $workspace_id,
        $data['name'], 
        $phone, 
        $data['email'] ?? null, 
        $data['facebook_id'] ?? null, 
        $data['notes'] ?? null
    ]);
    return $pdo->lastInsertId();
}

function getCustomerById($customer_id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = ?");
    $stmt->execute([$customer_id]);
    return $stmt->fetch();
}

// Sohbet işlemleri
function getActiveConversations($user_id, $workspace_id = null) {
    global $pdo;
    
    // Kullanıcı rolünü kontrol et
    $user = getUserById($user_id);
    
    // Workspace ID yoksa session'dan al
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    // Check if workspace_id column exists
    $hasWorkspaceColumn = false;
    try {
        $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
        $hasWorkspaceColumn = ($stmt->rowCount() > 0);
    } catch (Exception $e) {
        // Column doesn't exist, continue without filtering
    }
    
    if (in_array($user['role'], ['admin', 'supervisor'])) {
        // Admin ve Supervisor workspace'teki tüm sohbetleri görür
        $sql = "
            SELECT c.*, cu.name as customer_name, cu.phone as customer_phone, cu.facebook_id, cu.instagram_id,
                   (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
                   (SELECT created_at FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time,
                   (SELECT COUNT(*) FROM messages WHERE conversation_id = c.id AND sender_type = 'customer' AND is_read = 0) as unread_count
            FROM conversations c
            LEFT JOIN customers cu ON c.customer_id = cu.id
            WHERE c.status = 'active'
        ";
        
        // Add workspace filter if column exists
        if ($hasWorkspaceColumn && $workspace_id) {
            $sql .= " AND c.workspace_id = ?";
        }
        
        $sql .= " ORDER BY last_message_time DESC";
        
        $stmt = $pdo->prepare($sql);
        if ($hasWorkspaceColumn && $workspace_id) {
            $stmt->execute([$workspace_id]);
        } else {
            $stmt->execute();
        }
    } else {
        // Agent sadece kendisine atananları görür (workspace içinde)
        $sql = "
            SELECT c.*, cu.name as customer_name, cu.phone as customer_phone, cu.facebook_id, cu.instagram_id,
                   (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
                   (SELECT created_at FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time,
                   (SELECT COUNT(*) FROM messages WHERE conversation_id = c.id AND sender_type = 'customer' AND is_read = 0) as unread_count
            FROM conversations c
            LEFT JOIN customers cu ON c.customer_id = cu.id
            WHERE c.assigned_to = ? AND c.status = 'active'
        ";
        
        // Add workspace filter if column exists
        if ($hasWorkspaceColumn && $workspace_id) {
            $sql .= " AND c.workspace_id = ?";
        }
        
        $sql .= " ORDER BY last_message_time DESC";
        
        $stmt = $pdo->prepare($sql);
        if ($hasWorkspaceColumn && $workspace_id) {
            $stmt->execute([$user_id, $workspace_id]);
        } else {
            $stmt->execute([$user_id]);
        }
    }
    
    return $stmt->fetchAll();
}

function getConversationById($conversation_id) {
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT c.*, cu.name as customer_name, cu.phone as customer_phone, cu.email as customer_email
        FROM conversations c
        LEFT JOIN customers cu ON c.customer_id = cu.id
        WHERE c.id = ?
    ");
    $stmt->execute([$conversation_id]);
    return $stmt->fetch();
}

function createConversation($data) {
    global $pdo;
    
    // Auto-detect workspace from session if not in data
    $workspace_id = $data['workspace_id'] ?? null;
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    // Default to workspace 1 if still not set (backward compatibility)
    if (!$workspace_id) {
        $workspace_id = 1;
    }
    
    // Eğer array değilse eski format için dönüştür
    if (!is_array($data)) {
        $data = [
            'customer_id' => $data,
            'assigned_to' => func_get_arg(1),
            'status' => 'active',
            'platform' => 'whatsapp'
        ];
    }
    
    $stmt = $pdo->prepare("INSERT INTO conversations (workspace_id, customer_id, assigned_to, status, platform, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
    $stmt->execute([
        $workspace_id,
        $data['customer_id'],
        $data['assigned_to'],
        $data['status'] ?? 'active',
        $data['platform'] ?? 'whatsapp'
    ]);
    return $pdo->lastInsertId();
}

function getConversationByCustomer($customer_id, $platform = 'whatsapp') {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM conversations WHERE customer_id = ? AND status = 'active' AND (platform = ? OR platform IS NULL)");
    $stmt->execute([$customer_id, $platform]);
    return $stmt->fetch();
}

// Mesaj işlemleri
function getMessagesByConversation($conversation_id, $limit = 10000) {
    global $pdo;
    
    // Önce toplam mesaj sayısını kontrol et
    $countStmt = $pdo->prepare("SELECT COUNT(*) FROM messages WHERE conversation_id = ?");
    $countStmt->execute([$conversation_id]);
    $totalMessages = $countStmt->fetchColumn();
    
    // Eğer 10000'den fazla mesaj varsa uyarı logla (ama yine de getir)
    if ($totalMessages > 10000) {
        writeLog("Conversation $conversation_id has $totalMessages messages - consider implementing pagination", 'WARNING');
    }
    
    $stmt = $pdo->prepare("
        SELECT * FROM messages 
        WHERE conversation_id = ? 
        ORDER BY created_at ASC 
        LIMIT ?
    ");
    $stmt->execute([$conversation_id, $limit]);
    return $stmt->fetchAll();
}

function saveMessage($data) {
    global $pdo;
    
    // Eğer array değilse eski format için dönüştür
    if (!is_array($data)) {
        $data = [
            'conversation_id' => $data,
            'sender_id' => func_get_arg(1),
            'content' => func_get_arg(2),
            'sender_type' => func_get_arg(3) ?? 'customer',
            'whatsapp_message_id' => func_get_arg(4) ?? null
        ];
    }
    
    // Yeni format için content ve sender_type kullan
    $content = $data['content'] ?? $data['message'] ?? '';
    $senderType = $data['sender_type'] ?? ($data['message_type'] === 'sent' ? 'agent' : 'customer');
    $externalId = $data['external_id'] ?? $data['whatsapp_message_id'] ?? null;
    
    $stmt = $pdo->prepare("INSERT INTO messages (conversation_id, content, sender_type, sender_id, whatsapp_message_id, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
    $stmt->execute([
        $data['conversation_id'],
        $content,
        $senderType,
        $data['sender_id'] ?? null,
        $externalId
    ]);
    return $pdo->lastInsertId();
}

function markMessagesAsRead($conversation_id, $user_id) {
    global $pdo;
    $stmt = $pdo->prepare("
        UPDATE messages 
        SET is_read = 1, read_at = NOW(), read_by = ?
        WHERE conversation_id = ? AND sender_type = 'customer' AND is_read = 0
    ");
    return $stmt->execute([$user_id, $conversation_id]);
}

// Yardımcı fonksiyonlar
function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'Az önce';
    if ($time < 3600) return floor($time/60) . ' dakika önce';
    if ($time < 86400) return floor($time/3600) . ' saat önce';
    if ($time < 2592000) return floor($time/86400) . ' gün önce';
    
    return date('d.m.Y', strtotime($datetime));
}

function formatPhoneNumber($phone) {
    // Telefon numarasını temizle ve formatla
    $phone = preg_replace('/[^0-9+]/', '', $phone);
    
    // Türkiye numarası ise +90 ekle
    if (strlen($phone) == 10 && substr($phone, 0, 1) == '5') {
        $phone = '+90' . $phone;
    }
    
    return $phone;
}

function validatePhoneNumber($phone) {
    $phone = formatPhoneNumber($phone);
    return preg_match('/^\+90[5][0-9]{9}$/', $phone);
}

function sanitizeInput($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function generateRandomString($length = 10) {
    return substr(str_shuffle(str_repeat($x='0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length/strlen($x)) )),1,$length);
}

// WhatsApp webhook doğrulama
function verifyWebhook($token, $challenge) {
    global $whatsapp_config;
    return $token === $whatsapp_config['verify_token'];
}

// Log fonksiyonu
function writeLog($message, $level = 'INFO') {
    $logDir = 'logs';
    $logFile = $logDir . '/app_' . date('Y-m-d') . '.log';
    
    // Log dizinini oluştur
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logMessage = '[' . date('Y-m-d H:i:s') . '] [' . $level . '] ' . $message . PHP_EOL;
    
    // Log dosyasına yaz
    try {
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    } catch (Exception $e) {
        // Log yazma hatası durumunda sessizce devam et
        error_log("Log yazma hatası: " . $e->getMessage());
    }
}

// Facebook entegrasyonu için
/**
 * Get customer by Facebook ID
 */
function getCustomerByFacebookId($facebook_id, $workspace_id = null) {
    global $pdo;
    
    // Auto-detect workspace from session if not provided
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    if ($workspace_id) {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE facebook_id = ? AND workspace_id = ?");
        $stmt->execute([$facebook_id, $workspace_id]);
    } else {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE facebook_id = ?");
        $stmt->execute([$facebook_id]);
    }
    
    return $stmt->fetch();
}

/**
 * Get customer by Instagram ID
 */
function getCustomerByInstagramId($instagram_id, $workspace_id = null) {
    global $pdo;
    
    // Auto-detect workspace from session if not provided
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    if ($workspace_id) {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE instagram_id = ? AND workspace_id = ?");
        $stmt->execute([$instagram_id, $workspace_id]);
    } else {
        $stmt = $pdo->prepare("SELECT * FROM customers WHERE instagram_id = ?");
        $stmt->execute([$instagram_id]);
    }
    
    return $stmt->fetch();
}


function getConversationByCustomerIdAndPlatform($customer_id, $platform) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM conversations WHERE customer_id = ? AND platform = ?");
    $stmt->execute([$customer_id, $platform]);
    return $stmt->fetch();
}



function getFacebookUserProfile($user_id, $access_token) {
    $url = "https://graph.facebook.com/v18.0/{$user_id}?fields=first_name,last_name&access_token={$access_token}";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        return json_decode($response, true);
    }
    
    return ['first_name' => 'Facebook', 'last_name' => 'User'];
}


function updateConversation($conversation_id, $data) {
    global $pdo;
    $fields = [];
    $values = [];
    
    foreach ($data as $key => $value) {
        $fields[] = "{$key} = ?";
        $values[] = $value;
    }
    
    $values[] = $conversation_id;
    $sql = "UPDATE conversations SET " . implode(', ', $fields) . " WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    return $stmt->execute($values);
}


function log_action($user_id, $action, $details) {
    global $pdo;
    $stmt = $pdo->prepare("INSERT INTO activity_logs (user_id, action, details, created_at) VALUES (?, ?, ?, NOW())");
    return $stmt->execute([$user_id, $action, $details]);
}

// ============================================================================
// NEW FUNCTIONS FOR MULTI-CHANNEL & ASSIGNMENT SYSTEM
// ============================================================================

/**
 * Get conversations by platform
 */
function getConversationsByPlatform($user_id, $platform = null, $status = 'active', $workspace_id = null) {
    global $pdo;
    
    // Workspace ID yoksa session'dan al
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    // Check if workspace_id column exists
    $hasWorkspaceColumn = false;
    try {
        $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
        $hasWorkspaceColumn = ($stmt->rowCount() > 0);
    } catch (Exception $e) {
        // Column doesn't exist
    }
    
    $sql = "
        SELECT c.*, cu.name as customer_name, cu.phone as customer_phone, cu.facebook_id,
               (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
               (SELECT created_at FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time,
               (SELECT COUNT(*) FROM messages WHERE conversation_id = c.id AND sender_type = 'customer' AND is_read = 0) as unread_count
        FROM conversations c
        LEFT JOIN customers cu ON c.customer_id = cu.id
        WHERE c.assigned_to = ? AND c.status = ?
    ";
    
    $params = [$user_id, $status];
    
    // Add workspace filter if column exists
    if ($hasWorkspaceColumn && $workspace_id) {
        $sql .= " AND c.workspace_id = ?";
        $params[] = $workspace_id;
    }
    
    if ($platform) {
        $sql .= " AND c.platform = ?";
        $params[] = $platform;
    }
    
    $sql .= " ORDER BY last_message_time DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    return $stmt->fetchAll();
}

/**
 * Get all conversations (for admin/supervisor)
 */
function getAllConversations($platform = null, $status = 'active', $limit = 100, $workspace_id = null) {
    global $pdo;
    
    // Workspace ID yoksa session'dan al
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    // Check if workspace_id column exists
    $hasWorkspaceColumn = false;
    try {
        $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
        $hasWorkspaceColumn = ($stmt->rowCount() > 0);
    } catch (Exception $e) {
        // Column doesn't exist
    }
    
    $sql = "
        SELECT c.*, cu.name as customer_name, cu.phone as customer_phone, cu.facebook_id,
               u.name as agent_name,
               (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
               (SELECT created_at FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time,
               (SELECT COUNT(*) FROM messages WHERE conversation_id = c.id AND sender_type = 'customer' AND is_read = 0) as unread_count
        FROM conversations c
        LEFT JOIN customers cu ON c.customer_id = cu.id
        LEFT JOIN users u ON c.assigned_to = u.id
        WHERE c.status = ?
    ";
    
    $params = [$status];
    
    // Add workspace filter if column exists
    if ($hasWorkspaceColumn && $workspace_id) {
        $sql .= " AND c.workspace_id = ?";
        $params[] = $workspace_id;
    }
    
    if ($platform) {
        $sql .= " AND c.platform = ?";
        $params[] = $platform;
    }
    
    $sql .= " ORDER BY last_message_time DESC LIMIT ?";
    $params[] = $limit;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    return $stmt->fetchAll();
}

/**
 * Get unassigned conversations
 */
function getUnassignedConversations($platform = null, $limit = 50, $workspace_id = null) {
    global $pdo;
    
    // Workspace ID yoksa session'dan al
    if ($workspace_id === null && isset($_SESSION['workspace_id'])) {
        $workspace_id = $_SESSION['workspace_id'];
    }
    
    // Check if workspace_id column exists
    $hasWorkspaceColumn = false;
    try {
        $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
        $hasWorkspaceColumn = ($stmt->rowCount() > 0);
    } catch (Exception $e) {
        // Column doesn't exist
    }
    
    $sql = "
        SELECT c.*, cu.name as customer_name, cu.phone, cu.facebook_id,
               (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
               (SELECT created_at FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time
        FROM conversations c
        LEFT JOIN customers cu ON c.customer_id = cu.id
        WHERE (c.assigned_to IS NULL OR c.assigned_to = 0)
        AND c.status = 'active'
    ";
    
    $params = [];
    
    // Add workspace filter if column exists
    if ($hasWorkspaceColumn && $workspace_id) {
        $sql .= " AND c.workspace_id = ?";
        $params[] = $workspace_id;
    }
    
    if ($platform) {
        $sql .= " AND c.platform = ?";
        $params[] = $platform;
    }
    
    $sql .= " ORDER BY c.created_at ASC LIMIT ?";
    $params[] = $limit;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    return $stmt->fetchAll();
}

/**
 * Check user permission
 */
function checkUserPermission($user_id, $permission, $resource = null) {
    global $pdo;
    
    // Admin has all permissions
    $user = getUserById($user_id);
    if ($user && $user['role'] === 'admin') {
        return true;
    }
    
    // Check specific permission
    $sql = "SELECT COUNT(*) FROM user_permissions WHERE user_id = ? AND permission = ?";
    
    if ($resource) {
        $sql .= " AND (resource = ? OR resource IS NULL)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$user_id, $permission, $resource]);
    } else {
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$user_id, $permission]);
    }
    
    return $stmt->fetchColumn() > 0;
}

/**
 * Grant user permission
 */
function grantUserPermission($user_id, $permission, $resource = null, $granted_by = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO user_permissions (user_id, permission, resource, granted_by) 
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE granted_by = ?
        ");
        return $stmt->execute([$user_id, $permission, $resource, $granted_by, $granted_by]);
    } catch (Exception $e) {
        writeLog("Permission grant error: " . $e->getMessage(), 'ERROR');
        return false;
    }
}

/**
 * Get agent statistics
 */
function getAgentStatistics($user_id, $date = null, $platform = 'all') {
    global $pdo;
    
    if (!$date) {
        $date = date('Y-m-d');
    }
    
    $stmt = $pdo->prepare("
        SELECT * FROM agent_statistics 
        WHERE user_id = ? AND date = ? AND platform = ?
    ");
    $stmt->execute([$user_id, $date, $platform]);
    
    return $stmt->fetch();
}

/**
 * Update agent statistics
 */
function updateAgentStatistics($user_id, $data, $date = null, $platform = 'all') {
    global $pdo;
    
    if (!$date) {
        $date = date('Y-m-d');
    }
    
    // Build update fields
    $fields = [];
    $values = [];
    
    foreach ($data as $key => $value) {
        $fields[] = "$key = ?";
        $values[] = $value;
    }
    
    $values[] = $user_id;
    $values[] = $date;
    $values[] = $platform;
    
    try {
        // Try to update first
        $sql = "UPDATE agent_statistics SET " . implode(', ', $fields) . " WHERE user_id = ? AND date = ? AND platform = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($values);
        
        // If no rows affected, insert new record
        if ($stmt->rowCount() === 0) {
            $insertFields = array_keys($data);
            $insertFields[] = 'user_id';
            $insertFields[] = 'date';
            $insertFields[] = 'platform';
            
            $insertValues = array_values($data);
            $insertValues[] = $user_id;
            $insertValues[] = $date;
            $insertValues[] = $platform;
            
            $placeholders = str_repeat('?,', count($insertValues) - 1) . '?';
            $sql = "INSERT INTO agent_statistics (" . implode(',', $insertFields) . ") VALUES ($placeholders)";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($insertValues);
        }
        
        return true;
    } catch (Exception $e) {
        writeLog("Statistics update error: " . $e->getMessage(), 'ERROR');
        return false;
    }
}

/**
 * Get agent availability
 */
function getAgentAvailability($user_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT * FROM agent_availability WHERE user_id = ?");
    $stmt->execute([$user_id]);
    
    return $stmt->fetch();
}

/**
 * Update agent availability
 */
function updateAgentAvailability($user_id, $status, $auto_accept = null) {
    global $pdo;
    
    try {
        if ($auto_accept !== null) {
            $stmt = $pdo->prepare("
                UPDATE agent_availability 
                SET status = ?, auto_accept = ?, last_activity_at = NOW() 
                WHERE user_id = ?
            ");
            return $stmt->execute([$status, $auto_accept, $user_id]);
        } else {
            $stmt = $pdo->prepare("
                UPDATE agent_availability 
                SET status = ?, last_activity_at = NOW() 
                WHERE user_id = ?
            ");
            return $stmt->execute([$status, $user_id]);
        }
    } catch (Exception $e) {
        writeLog("Availability update error: " . $e->getMessage(), 'ERROR');
        return false;
    }
}

/**
 * Get platform icon class
 */
function getPlatformIcon($platform) {
    $icons = [
        'whatsapp' => 'fab fa-whatsapp',
        'facebook' => 'fab fa-facebook-messenger',
        'instagram' => 'fab fa-instagram'
    ];
    
    return $icons[$platform] ?? 'fas fa-comment';
}

/**
 * Get platform color
 */
function getPlatformColor($platform) {
    $colors = [
        'whatsapp' => '#25D366',
        'facebook' => '#0084FF',
        'instagram' => '#E4405F'
    ];
    
    return $colors[$platform] ?? '#6B7280';
}

/**
 * Get conversation statistics
 */
function getConversationStats($conversation_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_messages,
            SUM(CASE WHEN sender_type = 'agent' THEN 1 ELSE 0 END) as agent_messages,
            SUM(CASE WHEN sender_type = 'customer' THEN 1 ELSE 0 END) as customer_messages,
            MIN(created_at) as first_message_at,
            MAX(created_at) as last_message_at
        FROM messages
        WHERE conversation_id = ?
    ");
    $stmt->execute([$conversation_id]);
    
    return $stmt->fetch();
}


/**
 * Zamanlama ayarlarına göre AI'yı otomatik aktif/pasif et
 * Returns: true = AI aktif edilmeli, false = AI pasif edilmeli, null = zamanlama yok
 */
function checkAndUpdateAISchedule($conversation_id) {
    global $pdo;
    
    // Sohbetin chatbot'unu al
    $stmt = $pdo->prepare("
        SELECT c.chatbot_id, c.is_ai_active, cb.schedule_enabled, cb.schedule_start_time, 
               cb.schedule_end_time, cb.schedule_days
        FROM conversations c
        LEFT JOIN chatbots cb ON c.chatbot_id = cb.id
        WHERE c.id = ? AND cb.is_active = 1
    ");
    $stmt->execute([$conversation_id]);
    $data = $stmt->fetch();
    
    if (!$data) {
        return null; // Chatbot atanmamış
    }
    
    // Zamanlama aktif değilse, AI'yı her zaman aktif tut
    if (!$data['schedule_enabled']) {
        if (!$data['is_ai_active']) {
            // AI pasifse aktif et
            $stmt = $pdo->prepare("UPDATE conversations SET is_ai_active = 1 WHERE id = ?");
            $stmt->execute([$conversation_id]);
            return true;
        }
        return null; // Zaten aktif, değişiklik yok
    }
    
    // Türkiye saati
    date_default_timezone_set('Europe/Istanbul');
    $current_time = date('H:i:s');
    $current_day = date('w'); // 0 (Pazar) - 6 (Cumartesi)
    
    // Gün kontrolü
    $is_active_day = true;
    if ($data['schedule_days']) {
        $active_days = explode(',', $data['schedule_days']);
        $is_active_day = in_array($current_day, $active_days);
    }
    
    // Saat kontrolü
    $start_time = $data['schedule_start_time'];
    $end_time = $data['schedule_end_time'];
    
    if (!$start_time || !$end_time) {
        return null;
    }
    
    $is_within_time = false;
    
    // Gece geçişi kontrolü (örn: 22:00 - 08:00)
    if ($start_time > $end_time) {
        // Gece geçişi var
        $is_within_time = ($current_time >= $start_time || $current_time < $end_time);
    } else {
        // Normal aralık (örn: 09:00 - 18:00)
        $is_within_time = ($current_time >= $start_time && $current_time < $end_time);
    }
    
    $should_be_active = ($is_active_day && $is_within_time);
    
    // AI durumunu güncelle
    if ($should_be_active && !$data['is_ai_active']) {
        // AI'yı aktif et
        $stmt = $pdo->prepare("UPDATE conversations SET is_ai_active = 1 WHERE id = ?");
        $stmt->execute([$conversation_id]);
        return true;
    } else if (!$should_be_active && $data['is_ai_active']) {
        // AI'yı pasif et
        $stmt = $pdo->prepare("UPDATE conversations SET is_ai_active = 0 WHERE id = ?");
        $stmt->execute([$conversation_id]);
        return false;
    }
    
    return null; // Değişiklik yok
}


/**
 * Process AI Response using Google Gemini with Chatbot Support
 */
function processAIResponse($conversation_id, $message_content, $platform) {
    global $pdo, $google_ai_config;
    
    // 1. Sohbetin AI durumunu ve chatbot'unu al
    $stmt = $pdo->prepare("
        SELECT c.is_ai_active, c.customer_id, c.chatbot_id,
               cb.system_prompt, cb.model, cb.temperature, cb.max_tokens
        FROM conversations c
        LEFT JOIN chatbots cb ON c.chatbot_id = cb.id
        WHERE c.id = ?
    ");
    $stmt->execute([$conversation_id]);
    $conv = $stmt->fetch();
    
    if (!$conv || !$conv['is_ai_active']) {
        return false; // AI not active
    }
    
    // 2. Check if API key is set
    if (empty($google_ai_config['api_key'])) {
        error_log("Google AI API Key missing");
        return false;
    }
    
    // 3. Chatbot bilgi bankasını al
    $knowledge = "";
    if ($conv['chatbot_id']) {
        $stmt = $pdo->prepare("
            SELECT content FROM chatbot_knowledge 
            WHERE chatbot_id = ?
            ORDER BY created_at DESC
        ");
        $stmt->execute([$conv['chatbot_id']]);
        $files = $stmt->fetchAll();
        
        foreach ($files as $file) {
            if (!empty($file['content'])) {
                $knowledge .= $file['content'] . "\n\n";
            }
        }
    }
    
    // 4. Sistem prompt'unu belirle
    $system_prompt = $conv['system_prompt'] ?? $google_ai_config['system_instruction'] ?? 'Sen yardımsever bir müşteri temsilcisisin.';
    
    // Bilgi bankası varsa ekle
    if (!empty($knowledge)) {
        $system_prompt .= "\n\n=== BİLGİ BANKASI ===\n" . trim($knowledge) . "\n=== BİLGİ BANKASI SONU ===\n\nYukarıdaki bilgi bankasını kullanarak müşterinin sorularını yanıtla.";
    }
    
    // 5. Prompt oluştur
    $full_prompt = $system_prompt . "\n\nMüşteri: " . $message_content;
    
    $data = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $full_prompt]
                ]
            ]
        ]
    ];
    
    // Model ayarlarını ekle
    if ($conv['chatbot_id']) {
        $data['generationConfig'] = [
            'temperature' => floatval($conv['temperature'] ?? 0.7),
            'maxOutputTokens' => intval($conv['max_tokens'] ?? 1000)
        ];
    }
    
    // 6. Model adını belirle
    $model = $conv['model'] ?? $google_ai_config['model'] ?? 'gemini-2.0-flash';
    
    // Add -latest suffix if needed
    if (strpos($model, 'gemini-2.0') === false && strpos($model, '-latest') === false && strpos($model, 'gemini-pro') === false) {
        $model .= '-latest';
    }
    
    // 7. Call Google Gemini API
    $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key=" . $google_ai_config['api_key'];
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        error_log("Google AI API Error ($httpCode): " . $response);
        return false;
    }
    
    $result = json_decode($response, true);
    $ai_reply = $result['candidates'][0]['content']['parts'][0]['text'] ?? null;
    
    if ($ai_reply) {
        // 8. Send reply to customer
        $customer = getCustomerById($conv['customer_id']);
        $send_result = false;
        
        if ($platform === 'whatsapp') {
            $send_result = sendWhatsAppMessage($customer['phone'], $ai_reply);
        } else if ($platform === 'facebook') {
            $send_result = sendFacebookMessage($customer['facebook_id'], $ai_reply);
        } else if ($platform === 'instagram') {
            // Instagram uses Facebook API
            global $facebook_config;
            $url = "https://graph.facebook.com/{$facebook_config['api_version']}/me/messages?access_token={$facebook_config['page_access_token']}";
            $data = [
                'recipient' => ['id' => $customer['instagram_id']],
                'message' => ['text' => $ai_reply]
            ];
            
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $fb_response = curl_exec($ch);
            $send_result = (curl_getinfo($ch, CURLINFO_HTTP_CODE) === 200);
            curl_close($ch);
        }
        
        // 9. Save AI message to DB
        if ($send_result) {
            saveMessage([
                'conversation_id' => $conversation_id,
                'sender_id' => null,
                'content' => $ai_reply,
                'sender_type' => 'agent',
                'is_read' => 1
            ]);
            return true;
        }
    }
    
    return false;
}
?>
