<?php
/**
 * Workspace Management Functions
 * MultiCRM - Workspace/Agency Management
 */

/**
 * Get current workspace from session
 */
function getCurrentWorkspace() {
    if (!isset($_SESSION['workspace_id'])) {
        return null;
    }
    
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT a.*, ua.role as user_role
        FROM agencies a
        LEFT JOIN user_agencies ua ON a.id = ua.agency_id AND ua.user_id = ?
        WHERE a.id = ? AND a.status = 'active'
    ");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['workspace_id']]);
    return $stmt->fetch();
}

/**
 * Get all workspaces for a user
 */
function getUserWorkspaces($user_id) {
    global $pdo;
    
    // Try with is_default column first
    try {
        $stmt = $pdo->prepare("
            SELECT a.*, ua.role, ua.is_default
            FROM agencies a
            INNER JOIN user_agencies ua ON a.id = ua.agency_id
            WHERE ua.user_id = ? AND a.status = 'active'
            ORDER BY ua.is_default DESC, a.name ASC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        // Column doesn't exist yet, migration not run
        // Fallback without is_default
        $stmt = $pdo->prepare("
            SELECT a.*, ua.role, FALSE as is_default
            FROM agencies a
            INNER JOIN user_agencies ua ON a.id = ua.agency_id
            WHERE ua.user_id = ? AND a.status = 'active'
            ORDER BY a.name ASC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll();
    }
}

/**
 * Get default workspace for a user
 */
function getDefaultWorkspace($user_id) {
    global $pdo;
    
    // Try with is_default column first
    try {
        $stmt = $pdo->prepare("
            SELECT a.*, ua.role
            FROM agencies a
            INNER JOIN user_agencies ua ON a.id = ua.agency_id
            WHERE ua.user_id = ? AND ua.is_default = TRUE AND a.status = 'active'
            LIMIT 1
        ");
        $stmt->execute([$user_id]);
        $workspace = $stmt->fetch();
        
        if ($workspace) {
            return $workspace;
        }
    } catch (PDOException $e) {
        // Column doesn't exist yet, migration not run
    }
    
    // Fallback: get the first workspace
    $stmt = $pdo->prepare("
        SELECT a.*, ua.role
        FROM agencies a
        INNER JOIN user_agencies ua ON a.id = ua.agency_id
        WHERE ua.user_id = ? AND a.status = 'active'
        ORDER BY ua.created_at ASC
        LIMIT 1
    ");
    $stmt->execute([$user_id]);
    return $stmt->fetch();
}

/**
 * Switch to a different workspace
 */
function switchWorkspace($user_id, $workspace_id) {
    if (!hasWorkspaceAccess($user_id, $workspace_id)) {
        return false;
    }
    
    $_SESSION['workspace_id'] = $workspace_id;
    
    // Update workspace name in session
    $workspace = getCurrentWorkspace();
    if ($workspace) {
        $_SESSION['workspace_name'] = $workspace['name'];
        return true;
    }
    
    return false;
}

/**
 * Check if user has access to a workspace
 */
function hasWorkspaceAccess($user_id, $workspace_id) {
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count
        FROM user_agencies ua
        INNER JOIN agencies a ON ua.agency_id = a.id
        WHERE ua.user_id = ? AND ua.agency_id = ? AND a.status = 'active'
    ");
    $stmt->execute([$user_id, $workspace_id]);
    $result = $stmt->fetch();
    return $result['count'] > 0;
}

/**
 * Set default workspace for a user
 */
function setDefaultWorkspace($user_id, $workspace_id) {
    if (!hasWorkspaceAccess($user_id, $workspace_id)) {
        return false;
    }
    
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        
        // Remove default from all workspaces
        $stmt = $pdo->prepare("UPDATE user_agencies SET is_default = FALSE WHERE user_id = ?");
        $stmt->execute([$user_id]);
        
        // Set new default
        $stmt = $pdo->prepare("UPDATE user_agencies SET is_default = TRUE WHERE user_id = ? AND agency_id = ?");
        $stmt->execute([$user_id, $workspace_id]);
        
        $pdo->commit();
        return true;
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error setting default workspace: " . $e->getMessage());
        return false;
    }
}

/**
 * Get workspace settings
 */
function getWorkspaceSettings($workspace_id, $key = null) {
    global $pdo;
    
    if ($key) {
        $stmt = $pdo->prepare("SELECT value, type FROM workspace_settings WHERE workspace_id = ? AND `key` = ?");
        $stmt->execute([$workspace_id, $key]);
        $setting = $stmt->fetch();
        
        if (!$setting) {
            return null;
        }
        
        // Convert value based on type
        switch ($setting['type']) {
            case 'boolean':
                return (bool)$setting['value'];
            case 'number':
                return (int)$setting['value'];
            case 'json':
                return json_decode($setting['value'], true);
            default:
                return $setting['value'];
        }
    } else {
        $stmt = $pdo->prepare("SELECT `key`, value, type FROM workspace_settings WHERE workspace_id = ?");
        $stmt->execute([$workspace_id]);
        $settings = [];
        
        while ($row = $stmt->fetch()) {
            switch ($row['type']) {
                case 'boolean':
                    $settings[$row['key']] = (bool)$row['value'];
                    break;
                case 'number':
                    $settings[$row['key']] = (int)$row['value'];
                    break;
                case 'json':
                    $settings[$row['key']] = json_decode($row['value'], true);
                    break;
                default:
                    $settings[$row['key']] = $row['value'];
            }
        }
        
        return $settings;
    }
}

/**
 * Set workspace setting
 */
function setWorkspaceSetting($workspace_id, $key, $value, $type = 'string', $description = null) {
    global $pdo;
    
    // Convert value based on type
    switch ($type) {
        case 'boolean':
            $value = $value ? '1' : '0';
            break;
        case 'json':
            $value = json_encode($value);
            break;
        default:
            $value = (string)$value;
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO workspace_settings (workspace_id, `key`, value, type, description)
        VALUES (?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE value = ?, type = ?, description = ?
    ");
    
    return $stmt->execute([
        $workspace_id, $key, $value, $type, $description,
        $value, $type, $description
    ]);
}

/**
 * Create a new workspace
 */
function createWorkspace($name, $domain = null, $created_by_user_id = null) {
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        
        // Create workspace
        $stmt = $pdo->prepare("
            INSERT INTO agencies (name, domain, status, created_at)
            VALUES (?, ?, 'active', NOW())
        ");
        $stmt->execute([$name, $domain]);
        $workspace_id = $pdo->lastInsertId();
        
        // Add creator as owner if provided
        if ($created_by_user_id) {
            $stmt = $pdo->prepare("
                INSERT INTO user_agencies (user_id, agency_id, role, is_default, created_at)
                VALUES (?, ?, 'owner', FALSE, NOW())
            ");
            $stmt->execute([$created_by_user_id, $workspace_id]);
        }
        
        $pdo->commit();
        return $workspace_id;
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error creating workspace: " . $e->getMessage());
        return false;
    }
}

/**
 * Add user to workspace
 */
function addUserToWorkspace($user_id, $workspace_id, $role = 'agent') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO user_agencies (user_id, agency_id, role, is_default, created_at)
            VALUES (?, ?, ?, FALSE, NOW())
        ");
        return $stmt->execute([$user_id, $workspace_id, $role]);
    } catch (Exception $e) {
        error_log("Error adding user to workspace: " . $e->getMessage());
        return false;
    }
}

/**
 * Remove user from workspace
 */
function removeUserFromWorkspace($user_id, $workspace_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("DELETE FROM user_agencies WHERE user_id = ? AND agency_id = ?");
    return $stmt->execute([$user_id, $workspace_id]);
}

/**
 * Get workspace members
 */
function getWorkspaceMembers($workspace_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT u.*, ua.role, ua.is_default, ua.created_at as joined_at
        FROM users u
        INNER JOIN user_agencies ua ON u.id = ua.user_id
        WHERE ua.agency_id = ?
        ORDER BY ua.role ASC, u.name ASC
    ");
    $stmt->execute([$workspace_id]);
    return $stmt->fetchAll();
}

/**
 * Check if user is workspace owner or admin
 */
function isWorkspaceAdmin($user_id, $workspace_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT role FROM user_agencies 
        WHERE user_id = ? AND agency_id = ?
    ");
    $stmt->execute([$user_id, $workspace_id]);
    $result = $stmt->fetch();
    
    return $result && in_array($result['role'], ['owner', 'admin']);
}
?>
