-- MultiCRM Database Migration - Multi-Channel & Assignment System
-- Version: 2.0
-- Date: 2025-12-01
-- Description: Adds support for multi-channel messaging (WhatsApp, Facebook, Instagram),
--              automatic message assignment, user permissions, and activity tracking

-- ============================================================================
-- STEP 1: Update customers table for multi-channel support
-- ============================================================================

-- Add Facebook ID column if not exists
ALTER TABLE `customers` 
ADD COLUMN IF NOT EXISTS `facebook_id` VARCHAR(255) DEFAULT NULL AFTER `email`,
ADD COLUMN IF NOT EXISTS `instagram_id` VARCHAR(255) DEFAULT NULL AFTER `facebook_id`,
ADD COLUMN IF NOT EXISTS `preferred_channel` ENUM('whatsapp', 'facebook', 'instagram', 'any') DEFAULT 'any' AFTER `instagram_id`,
ADD COLUMN IF NOT EXISTS `last_seen_at` TIMESTAMP NULL DEFAULT NULL AFTER `preferred_channel`,
ADD COLUMN IF NOT EXISTS `language` VARCHAR(10) DEFAULT 'tr' AFTER `last_seen_at`;

-- Add indexes for performance
ALTER TABLE `customers`
ADD INDEX IF NOT EXISTS `idx_facebook_id` (`facebook_id`),
ADD INDEX IF NOT EXISTS `idx_instagram_id` (`instagram_id`),
ADD INDEX IF NOT EXISTS `idx_preferred_channel` (`preferred_channel`);

-- Update phone column to allow NULL (for Facebook/Instagram only customers)
ALTER TABLE `customers` 
MODIFY COLUMN `phone` VARCHAR(20) DEFAULT NULL;

-- Remove unique constraint from phone if exists (we'll use composite unique)
ALTER TABLE `customers` DROP INDEX IF EXISTS `phone`;

-- Add composite unique constraint
ALTER TABLE `customers`
ADD UNIQUE INDEX `idx_unique_customer` (`phone`, `facebook_id`, `instagram_id`);

-- ============================================================================
-- STEP 2: Update conversations table for platform and assignment tracking
-- ============================================================================

-- Add platform and assignment columns
ALTER TABLE `conversations`
ADD COLUMN IF NOT EXISTS `platform` ENUM('whatsapp', 'facebook', 'instagram') DEFAULT 'whatsapp' AFTER `status`,
ADD COLUMN IF NOT EXISTS `assignment_type` ENUM('manual', 'auto', 'round_robin', 'load_balanced') DEFAULT 'auto' AFTER `platform`,
ADD COLUMN IF NOT EXISTS `assigned_at` TIMESTAMP NULL DEFAULT NULL AFTER `assignment_type`,
ADD COLUMN IF NOT EXISTS `last_agent_message_at` TIMESTAMP NULL DEFAULT NULL AFTER `assigned_at`,
ADD COLUMN IF NOT EXISTS `last_customer_message_at` TIMESTAMP NULL DEFAULT NULL AFTER `last_agent_message_at`,
ADD COLUMN IF NOT EXISTS `first_response_time` INT DEFAULT NULL COMMENT 'Seconds' AFTER `last_customer_message_at`,
ADD COLUMN IF NOT EXISTS `avg_response_time` INT DEFAULT NULL COMMENT 'Seconds' AFTER `first_response_time`,
ADD COLUMN IF NOT EXISTS `message_count` INT DEFAULT 0 AFTER `avg_response_time`,
ADD COLUMN IF NOT EXISTS `agent_message_count` INT DEFAULT 0 AFTER `message_count`,
ADD COLUMN IF NOT EXISTS `customer_message_count` INT DEFAULT 0 AFTER `agent_message_count`;

-- Add indexes
ALTER TABLE `conversations`
ADD INDEX IF NOT EXISTS `idx_platform` (`platform`),
ADD INDEX IF NOT EXISTS `idx_assignment_type` (`assignment_type`),
ADD INDEX IF NOT EXISTS `idx_assigned_at` (`assigned_at`);

-- ============================================================================
-- STEP 3: Create assignment_rules table
-- ============================================================================

CREATE TABLE IF NOT EXISTS `assignment_rules` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(255) NOT NULL,
  `description` TEXT DEFAULT NULL,
  `platform` ENUM('whatsapp', 'facebook', 'instagram', 'all') DEFAULT 'all',
  `assignment_type` ENUM('round_robin', 'load_balanced', 'manual', 'skill_based') DEFAULT 'round_robin',
  `priority` INT DEFAULT 0 COMMENT 'Higher number = higher priority',
  `conditions` JSON DEFAULT NULL COMMENT 'Rule conditions',
  `target_agents` JSON DEFAULT NULL COMMENT 'Agent IDs or "all"',
  `max_conversations_per_agent` INT DEFAULT 10,
  `working_hours_only` TINYINT(1) DEFAULT 1,
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_platform` (`platform`),
  KEY `idx_is_active` (`is_active`),
  KEY `idx_priority` (`priority`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default assignment rule
INSERT INTO `assignment_rules` (`name`, `description`, `platform`, `assignment_type`, `priority`, `is_active`) VALUES
('Default Auto Assignment', 'Automatically assign new conversations using round-robin', 'all', 'round_robin', 1, 1);

-- ============================================================================
-- STEP 4: Create user_permissions table
-- ============================================================================

CREATE TABLE IF NOT EXISTS `user_permissions` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) NOT NULL,
  `permission` VARCHAR(100) NOT NULL,
  `resource` VARCHAR(100) DEFAULT NULL COMMENT 'e.g., platform name, conversation_id',
  `granted_by` INT(11) DEFAULT NULL,
  `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `idx_user_permission` (`user_id`, `permission`, `resource`),
  KEY `idx_permission` (`permission`),
  FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`granted_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- STEP 5: Create activity_logs table
-- ============================================================================

CREATE TABLE IF NOT EXISTS `activity_logs` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) DEFAULT NULL,
  `action` VARCHAR(100) NOT NULL,
  `resource_type` VARCHAR(50) DEFAULT NULL COMMENT 'conversation, customer, user, etc.',
  `resource_id` INT(11) DEFAULT NULL,
  `details` JSON DEFAULT NULL,
  `ip_address` VARCHAR(45) DEFAULT NULL,
  `user_agent` VARCHAR(500) DEFAULT NULL,
  `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_action` (`action`),
  KEY `idx_resource` (`resource_type`, `resource_id`),
  KEY `idx_created_at` (`created_at`),
  FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- STEP 6: Create conversation_transfers table
-- ============================================================================

CREATE TABLE IF NOT EXISTS `conversation_transfers` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `conversation_id` INT(11) NOT NULL,
  `from_agent_id` INT(11) NOT NULL,
  `to_agent_id` INT(11) NOT NULL,
  `transferred_by` INT(11) DEFAULT NULL COMMENT 'Who initiated the transfer',
  `reason` TEXT DEFAULT NULL,
  `notes` TEXT DEFAULT NULL,
  `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_conversation_id` (`conversation_id`),
  KEY `idx_from_agent` (`from_agent_id`),
  KEY `idx_to_agent` (`to_agent_id`),
  KEY `idx_created_at` (`created_at`),
  FOREIGN KEY (`conversation_id`) REFERENCES `conversations` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`from_agent_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`to_agent_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`transferred_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- STEP 7: Create agent_availability table
-- ============================================================================

CREATE TABLE IF NOT EXISTS `agent_availability` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) NOT NULL,
  `status` ENUM('online', 'offline', 'away', 'busy') DEFAULT 'offline',
  `max_conversations` INT DEFAULT 10,
  `current_conversations` INT DEFAULT 0,
  `auto_accept` TINYINT(1) DEFAULT 1,
  `working_hours` JSON DEFAULT NULL COMMENT 'Weekly schedule',
  `last_activity_at` TIMESTAMP NULL DEFAULT NULL,
  `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `idx_user_id` (`user_id`),
  KEY `idx_status` (`status`),
  FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create availability records for existing users
INSERT INTO `agent_availability` (`user_id`, `status`, `max_conversations`, `auto_accept`)
SELECT `id`, 'online', 10, 1 FROM `users` WHERE `status` = 'active'
ON DUPLICATE KEY UPDATE `user_id` = `user_id`;

-- ============================================================================
-- STEP 8: Create agent_statistics table
-- ============================================================================

CREATE TABLE IF NOT EXISTS `agent_statistics` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) NOT NULL,
  `date` DATE NOT NULL,
  `platform` ENUM('whatsapp', 'facebook', 'instagram', 'all') DEFAULT 'all',
  `conversations_handled` INT DEFAULT 0,
  `messages_sent` INT DEFAULT 0,
  `messages_received` INT DEFAULT 0,
  `avg_first_response_time` INT DEFAULT NULL COMMENT 'Seconds',
  `avg_response_time` INT DEFAULT NULL COMMENT 'Seconds',
  `conversations_resolved` INT DEFAULT 0,
  `conversations_transferred` INT DEFAULT 0,
  `customer_satisfaction_score` DECIMAL(3,2) DEFAULT NULL,
  `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `idx_user_date_platform` (`user_id`, `date`, `platform`),
  KEY `idx_date` (`date`),
  KEY `idx_platform` (`platform`),
  FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- STEP 9: Add new settings
-- ============================================================================

INSERT INTO `settings` (`key`, `value`, `type`, `description`) VALUES
('auto_assignment_enabled', '1', 'boolean', 'Enable automatic conversation assignment'),
('assignment_algorithm', 'round_robin', 'string', 'Assignment algorithm: round_robin, load_balanced'),
('max_conversations_per_agent', '10', 'number', 'Maximum conversations per agent'),
('working_hours_start', '09:00', 'string', 'Working hours start time'),
('working_hours_end', '18:00', 'string', 'Working hours end time'),
('working_days', 'monday,tuesday,wednesday,thursday,friday', 'string', 'Working days'),
('facebook_enabled', '1', 'boolean', 'Enable Facebook Messenger'),
('instagram_enabled', '0', 'boolean', 'Enable Instagram Direct Messages'),
('unified_inbox_enabled', '1', 'boolean', 'Enable unified inbox view')
ON DUPLICATE KEY UPDATE `key` = `key`;

-- ============================================================================
-- STEP 10: Update existing data
-- ============================================================================

-- Set platform for existing conversations (default to whatsapp)
UPDATE `conversations` SET `platform` = 'whatsapp' WHERE `platform` IS NULL;

-- Set assignment_type for existing conversations
UPDATE `conversations` SET `assignment_type` = 'manual' WHERE `assignment_type` IS NULL;

-- Set assigned_at for existing active conversations
UPDATE `conversations` SET `assigned_at` = `created_at` WHERE `assigned_at` IS NULL AND `status` = 'active';

-- ============================================================================
-- Migration Complete
-- ============================================================================

-- Log migration completion
INSERT INTO `activity_logs` (`action`, `resource_type`, `details`) VALUES
('database_migration', 'system', JSON_OBJECT('version', '2.0', 'description', 'Multi-channel and assignment system migration'));
