-- ============================================================================
-- MultiCRM - Chatwoot Style Multi-Tenant Architecture Migration
-- Version: 3.0 - Fixed
-- Date: 2025-12-03
-- Description: Inbox-based workspace isolation (Chatwoot model)
-- ============================================================================

-- ============================================================================
-- STEP 1: Create Inboxes Table (Core of Multi-Tenant)
-- ============================================================================

CREATE TABLE IF NOT EXISTS `inboxes` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `workspace_id` int(11) NOT NULL,
  `name` varchar(255) NOT NULL,
  `channel_type` enum('whatsapp','facebook','instagram','email','sms') NOT NULL,
  `identifier` varchar(100) NOT NULL,
  `phone_number_id` varchar(255) DEFAULT NULL,
  `business_account_id` varchar(255) DEFAULT NULL,
  `access_token` text DEFAULT NULL,
  `verify_token` varchar(255) DEFAULT NULL,
  `webhook_url` varchar(500) DEFAULT NULL,
  `settings` json DEFAULT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `identifier` (`identifier`),
  KEY `workspace_id` (`workspace_id`),
  KEY `channel_type` (`channel_type`),
  KEY `is_active` (`is_active`),
  FOREIGN KEY (`workspace_id`) REFERENCES `agencies` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- STEP 2: Add workspace_id to customers (if not exists)
-- ============================================================================

-- Add workspace_id column to customers
ALTER TABLE `customers` 
ADD COLUMN IF NOT EXISTS `workspace_id` INT NOT NULL DEFAULT 1 AFTER `id`;

-- Add index
ALTER TABLE `customers` 
ADD INDEX IF NOT EXISTS `idx_customers_workspace_id` (`workspace_id`);

-- ============================================================================
-- STEP 3: Add workspace_id to conversations (if not exists)
-- ============================================================================

-- Add workspace_id column to conversations
ALTER TABLE `conversations` 
ADD COLUMN IF NOT EXISTS `workspace_id` INT NOT NULL DEFAULT 1 AFTER `id`;

-- Add index
ALTER TABLE `conversations` 
ADD INDEX IF NOT EXISTS `idx_conversations_workspace_id` (`workspace_id`);

-- Add inbox_id column to conversations
ALTER TABLE `conversations` 
ADD COLUMN IF NOT EXISTS `inbox_id` INT NULL AFTER `workspace_id`;

-- Add index for inbox_id
ALTER TABLE `conversations` 
ADD INDEX IF NOT EXISTS `idx_conversations_inbox_id` (`inbox_id`);

-- ============================================================================
-- STEP 4: Update customers unique constraint for multi-tenant
-- ============================================================================

-- Drop old phone unique constraint if exists
ALTER TABLE `customers` DROP INDEX IF EXISTS `phone`;

-- Add composite unique: Same phone can exist in different workspaces
ALTER TABLE `customers` 
ADD UNIQUE INDEX IF NOT EXISTS `idx_unique_customer_phone` (`workspace_id`, `phone`);

-- Same for facebook_id (if column exists)
ALTER TABLE `customers` DROP INDEX IF EXISTS `facebook_id`;
ALTER TABLE `customers` 
ADD UNIQUE INDEX IF NOT EXISTS `idx_unique_customer_facebook` (`workspace_id`, `facebook_id`);

-- Same for instagram_id (if column exists)
ALTER TABLE `customers` DROP INDEX IF EXISTS `instagram_id`;
ALTER TABLE `customers` 
ADD UNIQUE INDEX IF NOT EXISTS `idx_unique_customer_instagram` (`workspace_id`, `instagram_id`);

-- ============================================================================
-- STEP 5: Add chatbots workspace_id (if table exists)
-- ============================================================================

ALTER TABLE `chatbots` 
ADD COLUMN IF NOT EXISTS `workspace_id` INT NOT NULL DEFAULT 1 AFTER `id`;

ALTER TABLE `chatbots` 
ADD INDEX IF NOT EXISTS `idx_chatbots_workspace_id` (`workspace_id`);

-- ============================================================================
-- STEP 6: Create inbox_members table (Who can access which inbox)
-- ============================================================================

CREATE TABLE IF NOT EXISTS `inbox_members` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `inbox_id` int(11) NOT NULL,
  `user_id` int(11) NOT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_inbox_member` (`inbox_id`, `user_id`),
  KEY `inbox_id` (`inbox_id`),
  KEY `user_id` (`user_id`),
  FOREIGN KEY (`inbox_id`) REFERENCES `inboxes` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- COMPLETED: Multi-Tenant Migration
-- ============================================================================
