<?php
/**
 * Multi-Tenant Migration Runner
 * Chatwoot-style Inbox Architecture
 */

session_start();
require_once '../config/database.php';
require_once '../includes/functions.php';

// Admin kontrolü
if (!isset($_SESSION['user_id'])) {
    die('Please login first: <a href="../login.php">Login</a>');
}

$user = getUserById($_SESSION['user_id']);
if ($user['role'] !== 'admin') {
    die('Only admin can run migrations');
}

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <title>Multi-Tenant Migration</title>
    <style>
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            padding: 20px; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        h1 { 
            color: #667eea; 
            border-bottom: 3px solid #667eea; 
            padding-bottom: 15px;
            margin-bottom: 30px;
        }
        .step { 
            background: #f8f9fa; 
            padding: 20px; 
            margin: 15px 0; 
            border-radius: 10px;
            border-left: 5px solid #667eea;
        }
        .success { color: #28a745; font-weight: bold; }
        .error { color: #dc3545; font-weight: bold; }
        .warning { color: #ffc107; font-weight: bold; }
        .info { color: #17a2b8; font-weight: bold; }
        pre { 
            background: #2d2d2d; 
            color: #f8f8f2;
            padding: 15px; 
            border-radius: 8px;
            overflow-x: auto;
            font-size: 13px;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: bold;
            transition: all 0.3s;
        }
        .btn:hover {
            background: #5568d3;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102,126,234,0.4);
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        table th {
            background: #667eea;
            color: white;
            padding: 12px;
            text-align: left;
        }
        table td {
            padding: 10px;
            border-bottom: 1px solid #ddd;
        }
        table tr:hover {
            background: #f8f9fa;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Multi-Tenant Migration (Chatwoot Model)</h1>
        
        <?php
        try {
            echo "<div class='step'>";
            echo "<h2>📊 Pre-Migration Check</h2>";
            
            // Check current state
            $checks = [];
            
            // Check if inboxes table exists
            $stmt = $pdo->query("SHOW TABLES LIKE 'inboxes'");
            $checks['inboxes_table'] = $stmt->rowCount() > 0;
            
            // Check workspace_id in customers
            $stmt = $pdo->query("SHOW COLUMNS FROM customers LIKE 'workspace_id'");
            $checks['customers_workspace_id'] = $stmt->rowCount() > 0;
            
            // Check workspace_id in conversations
            $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
            $checks['conversations_workspace_id'] = $stmt->rowCount() > 0;
            
            echo "<table>";
            echo "<tr><th>Check</th><th>Status</th></tr>";
            foreach ($checks as $check => $status) {
                $icon = $status ? '✅' : '❌';
                $class = $status ? 'success' : 'error';
                echo "<tr><td>{$check}</td><td class='{$class}'>{$icon} " . ($status ? 'OK' : 'Missing') . "</td></tr>";
            }
            echo "</table>";
            echo "</div>";
            
            // Run migration
            echo "<div class='step'>";
            echo "<h2>⚡ Running Migration...</h2>";
            
            $sql = file_get_contents(__DIR__ . '/multi-tenant-migration.sql');
            
            // Remove comments
            $sql = preg_replace('/--.*$/m', '', $sql);
            $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
            
            // Split by semicolon
            $statements = explode(';', $sql);
            
            $success_count = 0;
            $error_count = 0;
            $errors = [];
            $skipped = 0;
            
            foreach ($statements as $statement) {
                $statement = trim($statement);
                
                // Skip empty statements
                if (empty($statement)) {
                    continue;
                }
                
                try {
                    $pdo->exec($statement);
                    $success_count++;
                    
                    // Show what was executed
                    $preview = substr($statement, 0, 80);
                    echo "<div style='color: #28a745; font-size: 12px; margin: 2px 0;'>✅ " . htmlspecialchars($preview) . "...</div>";
                    
                } catch (PDOException $e) {
                    // Check if it's a "already exists" or "duplicate" error (these are OK)
                    if (strpos($e->getMessage(), 'Duplicate') !== false || 
                        strpos($e->getMessage(), 'already exists') !== false ||
                        strpos($e->getMessage(), "Can't DROP") !== false) {
                        $skipped++;
                        echo "<div style='color: #ffc107; font-size: 12px; margin: 2px 0;'>⚠️ Skipped (already exists)</div>";
                    } else {
                        $error_count++;
                        $errors[] = [
                            'sql' => substr($statement, 0, 150),
                            'error' => $e->getMessage()
                        ];
                        echo "<div style='color: #dc3545; font-size: 12px; margin: 2px 0;'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</div>";
                    }
                }
            }
            
            echo "<br>";
            echo "<p><span class='success'>✅ Successful: {$success_count}</span></p>";
            echo "<p><span class='warning'>⚠️ Skipped: {$skipped}</span></p>";
            echo "<p><span class='error'>❌ Errors: {$error_count}</span></p>";
            
            if (!empty($errors)) {
                echo "<h3>🔴 Critical Errors:</h3>";
                foreach ($errors as $error) {
                    echo "<div style='background: #f8d7da; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
                    echo "<strong>SQL:</strong><pre>" . htmlspecialchars($error['sql']) . "</pre>";
                    echo "<strong>Error:</strong><pre>" . htmlspecialchars($error['error']) . "</pre>";
                    echo "</div>";
                }
            }
            echo "</div>";
            
            // Post-migration check
            echo "<div class='step'>";
            echo "<h2>✅ Post-Migration Verification</h2>";
            
            // Inboxes count
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM inboxes");
            $inbox_count = $stmt->fetch()['count'];
            
            // Customers with workspace
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM customers WHERE workspace_id IS NOT NULL");
            $customer_count = $stmt->fetch()['count'];
            
            // Conversations with workspace
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM conversations WHERE workspace_id IS NOT NULL");
            $conversation_count = $stmt->fetch()['count'];
            
            echo "<table>";
            echo "<tr><th>Metric</th><th>Count</th></tr>";
            echo "<tr><td>Inboxes Created</td><td class='success'>{$inbox_count}</td></tr>";
            echo "<tr><td>Customers with Workspace</td><td class='info'>{$customer_count}</td></tr>";
            echo "<tr><td>Conversations with Workspace</td><td class='info'>{$conversation_count}</td></tr>";
            echo "</table>";
            echo "</div>";
            
            // Show created inboxes
            echo "<div class='step'>";
            echo "<h2>📥 Created Inboxes</h2>";
            
            $stmt = $pdo->query("
                SELECT i.*, a.name as workspace_name 
                FROM inboxes i 
                LEFT JOIN agencies a ON i.workspace_id = a.id
                ORDER BY i.created_at DESC
            ");
            $inboxes = $stmt->fetchAll();
            
            if ($inboxes) {
                echo "<table>";
                echo "<tr><th>ID</th><th>Workspace</th><th>Name</th><th>Type</th><th>Identifier</th><th>Status</th></tr>";
                foreach ($inboxes as $inbox) {
                    $status = $inbox['is_active'] ? '<span class="success">Active</span>' : '<span class="error">Inactive</span>';
                    echo "<tr>";
                    echo "<td>{$inbox['id']}</td>";
                    echo "<td>{$inbox['workspace_name']}</td>";
                    echo "<td>{$inbox['name']}</td>";
                    echo "<td>{$inbox['channel_type']}</td>";
                    echo "<td><code>{$inbox['identifier']}</code></td>";
                    echo "<td>{$status}</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p class='warning'>⚠️ No inboxes found. Will be created when you add channels.</p>";
            }
            
            echo "</div>";
            
            // Next steps
            echo "<div class='step' style='background: #d4edda; border-left-color: #28a745;'>";
            echo "<h2>🎯 Next Steps</h2>";
            echo "<ol style='font-size: 16px; line-height: 1.8;'>";
            echo "<li><strong>Inbox Management:</strong> Create inbox management page</li>";
            echo "<li><strong>Update Functions:</strong> Add workspace filters to all queries</li>";
            echo "<li><strong>Update Webhooks:</strong> Use inbox identifiers</li>";
            echo "<li><strong>Update Pages:</strong> Filter data by workspace</li>";
            echo "<li><strong>Test:</strong> Verify multi-tenant isolation</li>";
            echo "</ol>";
            echo "<br>";
            echo "<a href='../workspaces.php' class='btn'>Go to Workspaces</a> ";
            echo "<a href='../index.php' class='btn'>Go to Dashboard</a>";
            echo "</div>";
            
        } catch (Exception $e) {
            echo "<div class='step' style='background: #f8d7da; border-left-color: #dc3545;'>";
            echo "<h2>❌ Migration Failed</h2>";
            echo "<pre>" . htmlspecialchars($e->getMessage()) . "</pre>";
            echo "</div>";
        }
        ?>
    </div>
</body>
</html>

