<?php
/**
 * Workspace Migration Runner
 * Run this file once to add workspace functionality to the database
 */

session_start();
require_once '../config/database.php';
require_once '../includes/functions.php';

// Check if user is admin
if (!isset($_SESSION['user_id'])) {
    die('Please login first');
}

$user = getUserById($_SESSION['user_id']);
if ($user['role'] !== 'admin') {
    die('Only admins can run migrations');
}

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Workspace Migration</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background: #f5f7fa; padding: 40px 0; }
        .migration-card { background: white; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); padding: 30px; max-width: 800px; margin: 0 auto; }
        .success { color: #4caf50; }
        .error { color: #f44336; }
        .info { color: #2196f3; }
    </style>
</head>
<body>
    <div class="container">
        <div class="migration-card">
            <h2><i class="fas fa-database"></i> Workspace Migration</h2>
            <p>Bu script veritabanına workspace desteği ekleyecek.</p>
            <hr>
            
            <?php
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                echo "<h4>Migration başlatılıyor...</h4>";
                
                // Read SQL file
                $sqlFile = __DIR__ . '/workspace-migration.sql';
                if (!file_exists($sqlFile)) {
                    echo "<p class='error'>❌ Migration dosyası bulunamadı: $sqlFile</p>";
                    exit;
                }
                
                $sql = file_get_contents($sqlFile);
                
                // Remove comments
                $sql = preg_replace('/--.*$/m', '', $sql);
                
                // Split by semicolon but keep multi-line statements together
                $statements = explode(';', $sql);
                
                $successCount = 0;
                $errorCount = 0;
                
                foreach ($statements as $statement) {
                    $statement = trim($statement);
                    
                    // Skip empty statements
                    if (empty($statement)) {
                        continue;
                    }
                    
                    try {
                        $pdo->exec($statement);
                        $successCount++;
                        
                        // Extract table/column name for display
                        if (preg_match('/ALTER TABLE\s+(\w+)/i', $statement, $matches)) {
                            echo "<p class='success'>✓ Tablo güncellendi: {$matches[1]}</p>";
                        } elseif (preg_match('/CREATE TABLE.*?`?(\w+)`?/i', $statement, $matches)) {
                            echo "<p class='success'>✓ Tablo oluşturuldu: {$matches[1]}</p>";
                        } elseif (preg_match('/UPDATE\s+(\w+)/i', $statement, $matches)) {
                            echo "<p class='success'>✓ Veri güncellendi: {$matches[1]}</p>";
                        } else {
                            echo "<p class='success'>✓ SQL çalıştırıldı</p>";
                        }
                    } catch (PDOException $e) {
                        $errorCount++;
                        $errorMsg = $e->getMessage();
                        
                        // Ignore "Duplicate column" errors (already migrated)
                        if (strpos($errorMsg, 'Duplicate column') !== false || 
                            strpos($errorMsg, 'Duplicate key') !== false ||
                            strpos($errorMsg, 'already exists') !== false) {
                            echo "<p class='info'>ℹ️ Zaten mevcut (atlandı)</p>";
                            $errorCount--;
                        } else {
                            echo "<p class='error'>❌ Hata: " . htmlspecialchars($errorMsg) . "</p>";
                            echo "<p class='error'>SQL: " . htmlspecialchars(substr($statement, 0, 200)) . "...</p>";
                        }
                    }
                }
                
                echo "<hr>";
                echo "<h4>Migration Tamamlandı</h4>";
                echo "<p><strong>Başarılı:</strong> $successCount</p>";
                echo "<p><strong>Hata:</strong> $errorCount</p>";
                
                if ($errorCount === 0) {
                    echo "<div class='alert alert-success'>";
                    echo "<h5>✅ Migration başarıyla tamamlandı!</h5>";
                    echo "<p>Workspace sistemi aktif. Şimdi yapmanız gerekenler:</p>";
                    echo "<ol>";
                    echo "<li>Tüm mevcut müşterilerin workspace_id'sini güncelleyin (varsayılan: 1)</li>";
                    echo "<li>Tüm mevcut sohbetlerin workspace_id'sini güncelleyin (varsayılan: 1)</li>";
                    echo "<li><a href='../index.php'>Ana sayfaya dönün</a> ve workspace seçiciyi test edin</li>";
                    echo "</ol>";
                    echo "</div>";
                } else {
                    echo "<div class='alert alert-warning'>";
                    echo "<p>Bazı hatalar oluştu. Lütfen hataları kontrol edin.</p>";
                    echo "</div>";
                }
            } else {
            ?>
                <div class="alert alert-warning">
                    <h5>⚠️ Dikkat!</h5>
                    <p>Bu işlem veritabanınızı değiştirecek. Devam etmeden önce:</p>
                    <ul>
                        <li>Veritabanı yedeği aldığınızdan emin olun</li>
                        <li>Bu migration'ı sadece bir kez çalıştırın</li>
                        <li>Admin yetkileriniz olduğundan emin olun</li>
                    </ul>
                </div>
                
                <form method="POST">
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="checkbox" id="confirm" required>
                        <label class="form-check-label" for="confirm">
                            Veritabanı yedeği aldım ve migration'ı çalıştırmak istiyorum
                        </label>
                    </div>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-play"></i> Migration'ı Çalıştır
                    </button>
                    <a href="../index.php" class="btn btn-secondary">
                        <i class="fas fa-times"></i> İptal
                    </a>
                </form>
            <?php } ?>
        </div>
    </div>
    
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</body>
</html>
