<?php
/**
 * Simple Multi-Tenant Migration
 * Step-by-step with proper error handling
 */

session_start();
require_once '../config/database.php';
require_once '../includes/functions.php';

// Admin check
if (!isset($_SESSION['user_id'])) {
    die('Please <a href="../login.php">login</a> first');
}

$user = getUserById($_SESSION['user_id']);
if ($user['role'] !== 'admin') {
    die('Only admin can run migrations');
}

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Multi-Tenant Setup</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            padding: 40px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        h1 { 
            color: #667eea; 
            margin-bottom: 30px;
            font-size: 32px;
        }
        .step {
            background: #f8f9fa;
            padding: 20px;
            margin: 20px 0;
            border-radius: 10px;
            border-left: 5px solid #667eea;
        }
        .success { color: #28a745; font-weight: bold; }
        .error { color: #dc3545; font-weight: bold; }
        .warning { color: #ffc107; font-weight: bold; }
        .info { color: #17a2b8; }
        pre {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 15px;
            border-radius: 8px;
            overflow-x: auto;
            margin: 10px 0;
        }
        .btn {
            display: inline-block;
            padding: 12px 30px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            margin: 10px 5px;
            transition: all 0.3s;
        }
        .btn:hover {
            background: #5568d3;
            transform: translateY(-2px);
        }
        ul { margin: 15px 0; padding-left: 20px; }
        li { margin: 8px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Multi-Tenant Setup (Chatwoot Style)</h1>
        
        <?php
        $steps = [];
        $errors = [];
        
        try {
            // Step 1: Create inboxes table
            echo "<div class='step'>";
            echo "<h2>Step 1: Creating Inboxes Table</h2>";
            
            try {
                $pdo->exec("
                    CREATE TABLE IF NOT EXISTS `inboxes` (
                      `id` int(11) NOT NULL AUTO_INCREMENT,
                      `workspace_id` int(11) NOT NULL,
                      `name` varchar(255) NOT NULL,
                      `channel_type` enum('whatsapp','facebook','instagram','email','sms') NOT NULL,
                      `identifier` varchar(100) NOT NULL,
                      `phone_number_id` varchar(255) DEFAULT NULL,
                      `business_account_id` varchar(255) DEFAULT NULL,
                      `access_token` text DEFAULT NULL,
                      `verify_token` varchar(255) DEFAULT NULL,
                      `webhook_url` varchar(500) DEFAULT NULL,
                      `settings` json DEFAULT NULL,
                      `is_active` tinyint(1) DEFAULT 1,
                      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                      `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                      PRIMARY KEY (`id`),
                      UNIQUE KEY `identifier` (`identifier`),
                      KEY `workspace_id` (`workspace_id`),
                      KEY `channel_type` (`channel_type`),
                      FOREIGN KEY (`workspace_id`) REFERENCES `agencies` (`id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                ");
                echo "<p class='success'>✅ Inboxes table created</p>";
                $steps['inboxes_table'] = true;
            } catch (PDOException $e) {
                if (strpos($e->getMessage(), 'already exists') !== false) {
                    echo "<p class='warning'>⚠️ Inboxes table already exists (OK)</p>";
                    $steps['inboxes_table'] = true;
                } else {
                    throw $e;
                }
            }
            echo "</div>";
            
            // Step 2: Add workspace_id to customers
            echo "<div class='step'>";
            echo "<h2>Step 2: Updating Customers Table</h2>";
            
            // Check if column exists
            $stmt = $pdo->query("SHOW COLUMNS FROM customers LIKE 'workspace_id'");
            if ($stmt->rowCount() == 0) {
                $pdo->exec("ALTER TABLE customers ADD COLUMN workspace_id INT NOT NULL DEFAULT 1 AFTER id");
                $pdo->exec("ALTER TABLE customers ADD INDEX idx_customers_workspace (workspace_id)");
                echo "<p class='success'>✅ Added workspace_id to customers</p>";
            } else {
                echo "<p class='warning'>⚠️ workspace_id already exists in customers (OK)</p>";
            }
            $steps['customers_workspace'] = true;
            echo "</div>";
            
            // Step 3: Add workspace_id to conversations
            echo "<div class='step'>";
            echo "<h2>Step 3: Updating Conversations Table</h2>";
            
            $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'workspace_id'");
            if ($stmt->rowCount() == 0) {
                $pdo->exec("ALTER TABLE conversations ADD COLUMN workspace_id INT NOT NULL DEFAULT 1 AFTER id");
                $pdo->exec("ALTER TABLE conversations ADD INDEX idx_conversations_workspace (workspace_id)");
                echo "<p class='success'>✅ Added workspace_id to conversations</p>";
            } else {
                echo "<p class='warning'>⚠️ workspace_id already exists in conversations (OK)</p>";
            }
            
            // Add inbox_id
            $stmt = $pdo->query("SHOW COLUMNS FROM conversations LIKE 'inbox_id'");
            if ($stmt->rowCount() == 0) {
                $pdo->exec("ALTER TABLE conversations ADD COLUMN inbox_id INT NULL AFTER workspace_id");
                $pdo->exec("ALTER TABLE conversations ADD INDEX idx_conversations_inbox (inbox_id)");
                echo "<p class='success'>✅ Added inbox_id to conversations</p>";
            } else {
                echo "<p class='warning'>⚠️ inbox_id already exists in conversations (OK)</p>";
            }
            $steps['conversations_workspace'] = true;
            echo "</div>";
            
            // Step 4: Update unique constraints
            echo "<div class='step'>";
            echo "<h2>Step 4: Updating Unique Constraints</h2>";
            
            try {
                $pdo->exec("ALTER TABLE customers DROP INDEX IF EXISTS phone");
                echo "<p class='info'>ℹ️ Dropped old phone unique constraint</p>";
            } catch (PDOException $e) {
                // Ignore if doesn't exist
            }
            
            try {
                $pdo->exec("ALTER TABLE customers ADD UNIQUE INDEX idx_unique_customer_phone (workspace_id, phone)");
                echo "<p class='success'>✅ Added composite unique (workspace_id, phone)</p>";
            } catch (PDOException $e) {
                if (strpos($e->getMessage(), 'Duplicate') !== false) {
                    echo "<p class='warning'>⚠️ Unique constraint already exists (OK)</p>";
                } else {
                    echo "<p class='warning'>⚠️ " . $e->getMessage() . "</p>";
                }
            }
            $steps['unique_constraints'] = true;
            echo "</div>";
            
            // Step 5: Create inbox_members
            echo "<div class='step'>";
            echo "<h2>Step 5: Creating Inbox Members Table</h2>";
            
            try {
                $pdo->exec("
                    CREATE TABLE IF NOT EXISTS inbox_members (
                      id int(11) NOT NULL AUTO_INCREMENT,
                      inbox_id int(11) NOT NULL,
                      user_id int(11) NOT NULL,
                      created_at timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                      PRIMARY KEY (id),
                      UNIQUE KEY unique_inbox_member (inbox_id, user_id),
                      KEY inbox_id (inbox_id),
                      KEY user_id (user_id),
                      FOREIGN KEY (inbox_id) REFERENCES inboxes(id) ON DELETE CASCADE,
                      FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                ");
                echo "<p class='success'>✅ Inbox members table created</p>";
                $steps['inbox_members'] = true;
            } catch (PDOException $e) {
                if (strpos($e->getMessage(), 'already exists') !== false) {
                    echo "<p class='warning'>⚠️ Inbox members table already exists (OK)</p>";
                    $steps['inbox_members'] = true;
                } else {
                    throw $e;
                }
            }
            echo "</div>";
            
            // Step 6: Add chatbots workspace_id
            echo "<div class='step'>";
            echo "<h2>Step 6: Updating Chatbots Table</h2>";
            
            $stmt = $pdo->query("SHOW TABLES LIKE 'chatbots'");
            if ($stmt->rowCount() > 0) {
                $stmt = $pdo->query("SHOW COLUMNS FROM chatbots LIKE 'workspace_id'");
                if ($stmt->rowCount() == 0) {
                    $pdo->exec("ALTER TABLE chatbots ADD COLUMN workspace_id INT NOT NULL DEFAULT 1 AFTER id");
                    $pdo->exec("ALTER TABLE chatbots ADD INDEX idx_chatbots_workspace (workspace_id)");
                    echo "<p class='success'>✅ Added workspace_id to chatbots</p>";
                } else {
                    echo "<p class='warning'>⚠️ workspace_id already exists in chatbots (OK)</p>";
                }
            } else {
                echo "<p class='info'>ℹ️ Chatbots table not found (will be created when needed)</p>";
            }
            echo "</div>";
            
            // Summary
            echo "<div class='step' style='background: #d4edda; border-left-color: #28a745;'>";
            echo "<h2>✅ Migration Successful!</h2>";
            echo "<p><strong>Completed Steps:</strong></p>";
            echo "<ul>";
            foreach ($steps as $step => $status) {
                echo "<li class='success'>✅ " . ucfirst(str_replace('_', ' ', $step)) . "</li>";
            }
            echo "</ul>";
            
            // Show stats
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM inboxes");
            $inbox_count = $stmt->fetch()['count'];
            
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM customers WHERE workspace_id IS NOT NULL");
            $customer_count = $stmt->fetch()['count'];
            
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM conversations WHERE workspace_id IS NOT NULL");
            $conversation_count = $stmt->fetch()['count'];
            
            echo "<p style='margin-top: 20px;'><strong>Current Status:</strong></p>";
            echo "<ul>";
            echo "<li>📥 Inboxes: <span class='info'>{$inbox_count}</span></li>";
            echo "<li>👥 Customers with workspace: <span class='info'>{$customer_count}</span></li>";
            echo "<li>💬 Conversations with workspace: <span class='info'>{$conversation_count}</span></li>";
            echo "</ul>";
            
            echo "<div style='margin-top: 30px;'>";
            echo "<a href='../workspaces.php' class='btn'>📁 Go to Workspaces</a>";
            echo "<a href='../index.php' class='btn'>🏠 Go to Dashboard</a>";
            echo "</div>";
            echo "</div>";
            
        } catch (Exception $e) {
            echo "<div class='step' style='background: #f8d7da; border-left-color: #dc3545;'>";
            echo "<h2 class='error'>❌ Migration Failed</h2>";
            echo "<pre>" . htmlspecialchars($e->getMessage()) . "</pre>";
            echo "<p style='margin-top: 20px;'>Please contact support or check the error above.</p>";
            echo "</div>";
        }
        ?>
    </div>
</body>
</html>

