<?php
/**
 * Facebook OAuth Callback Handler
 * Handles OAuth responses from Facebook for both Facebook Messenger and Instagram
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/functions.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    die('Unauthorized: Please login first');
}

$user_id = $_SESSION['user_id'];
$workspace_id = $_SESSION['workspace_id'] ?? 1;

// Get Facebook App credentials
$stmt = $pdo->prepare("SELECT value FROM settings WHERE `key` = 'facebook_app_id' LIMIT 1");
$stmt->execute();
$result = $stmt->fetch();
$fb_app_id = $result ? $result['value'] : '';

$stmt = $pdo->prepare("SELECT value FROM settings WHERE `key` = 'facebook_app_secret' LIMIT 1");
$stmt->execute();
$result = $stmt->fetch();
$fb_app_secret = $result ? $result['value'] : '';

if (empty($fb_app_id) || empty($fb_app_secret)) {
    die('Error: Facebook App credentials not configured');
}

// Check for errors
if (isset($_GET['error'])) {
    $error_reason = $_GET['error_reason'] ?? 'Unknown';
    $error_description = $_GET['error_description'] ?? 'Unknown error';
    die("OAuth Error: {$error_reason} - {$error_description}");
}

// Get authorization code
$code = $_GET['code'] ?? '';
$state = $_GET['state'] ?? '';

if (empty($code)) {
    die('Error: No authorization code received');
}

// Determine channel type from state
$channel_type = 'facebook'; // default
if (strpos($state, 'instagram_') === 0) {
    $channel_type = 'instagram';
}

try {
    // Exchange code for access token
    $redirect_uri = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                    '://' . $_SERVER['HTTP_HOST'] . '/multicrm/oauth/facebook/callback.php';
    
    $token_url = "https://graph.facebook.com/v18.0/oauth/access_token?" . http_build_query([
        'client_id' => $fb_app_id,
        'client_secret' => $fb_app_secret,
        'redirect_uri' => $redirect_uri,
        'code' => $code
    ]);
    
    $ch = curl_init($token_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200) {
        throw new Exception("Failed to get access token. Response: " . $response);
    }
    
    $token_data = json_decode($response, true);
    $short_lived_token = $token_data['access_token'] ?? '';
    
    if (empty($short_lived_token)) {
        throw new Exception("No access token in response: " . $response);
    }
    
    // Exchange short-lived token for long-lived token
    $long_lived_url = "https://graph.facebook.com/v18.0/oauth/access_token?" . http_build_query([
        'grant_type' => 'fb_exchange_token',
        'client_id' => $fb_app_id,
        'client_secret' => $fb_app_secret,
        'fb_exchange_token' => $short_lived_token
    ]);
    
    $ch = curl_init($long_lived_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    curl_close($ch);
    
    $long_lived_data = json_decode($response, true);
    $access_token = $long_lived_data['access_token'] ?? $short_lived_token;
    
    // Get user's pages
    $pages_url = "https://graph.facebook.com/v18.0/me/accounts?access_token=" . urlencode($access_token);
    
    $ch = curl_init($pages_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    curl_close($ch);
    
    $pages_data = json_decode($response, true);
    $pages = $pages_data['data'] ?? [];
    
    if (empty($pages)) {
        throw new Exception("No Facebook Pages found. Please create a Facebook Page first.");
    }
    
    // For simplicity, use the first page
    // In production, you should let user select which page to connect
    $page = $pages[0];
    $page_id = $page['id'];
    $page_name = $page['name'];
    $page_access_token = $page['access_token'];
    
    // If connecting Instagram, get Instagram account connected to this page
    if ($channel_type === 'instagram') {
        $ig_url = "https://graph.facebook.com/v18.0/{$page_id}?fields=instagram_business_account&access_token=" . urlencode($page_access_token);
        
        $ch = curl_init($ig_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $response = curl_exec($ch);
        curl_close($ch);
        
        $ig_data = json_decode($response, true);
        $instagram_account_id = $ig_data['instagram_business_account']['id'] ?? null;
        
        if (!$instagram_account_id) {
            throw new Exception("No Instagram Business Account connected to this page. Please connect an Instagram Business Account to your Facebook Page first.");
        }
        
        // Get Instagram account details
        $ig_details_url = "https://graph.facebook.com/v18.0/{$instagram_account_id}?fields=id,username,name&access_token=" . urlencode($page_access_token);
        
        $ch = curl_init($ig_details_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $response = curl_exec($ch);
        curl_close($ch);
        
        $ig_details = json_decode($response, true);
        $identifier = '@' . ($ig_details['username'] ?? $instagram_account_id);
        $inbox_name = ($ig_details['name'] ?? 'Instagram') . ' - Direct Messages';
    } else {
        // Facebook Messenger
        $identifier = $page_id;
        $inbox_name = $page_name . ' - Messenger';
        $instagram_account_id = null;
    }
    
    // Generate webhook verify token
    $verify_token = bin2hex(random_bytes(16));
    
    // Save to inboxes table
    $stmt = $pdo->prepare("
        INSERT INTO inboxes (
            workspace_id, 
            name, 
            channel_type, 
            identifier, 
            page_id, 
            business_account_id,
            access_token, 
            verify_token,
            settings,
            is_active
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
        ON DUPLICATE KEY UPDATE
            access_token = VALUES(access_token),
            verify_token = VALUES(verify_token),
            updated_at = NOW()
    ");
    
    $settings = json_encode([
        'page_name' => $page_name,
        'instagram_account_id' => $instagram_account_id,
        'connected_at' => date('Y-m-d H:i:s'),
        'connected_by' => $user_id
    ]);
    
    $stmt->execute([
        $workspace_id,
        $inbox_name,
        $channel_type,
        $identifier,
        $page_id,
        $instagram_account_id,
        $page_access_token, // Should be encrypted in production
        $verify_token,
        $settings
    ]);
    
    $inbox_id = $pdo->lastInsertId();
    
    // Add current user as inbox member
    $stmt = $pdo->prepare("
        INSERT IGNORE INTO inbox_members (inbox_id, user_id) 
        VALUES (?, ?)
    ");
    $stmt->execute([$inbox_id, $user_id]);
    
    // Subscribe to webhooks
    $webhook_url = "https://graph.facebook.com/v18.0/{$page_id}/subscribed_apps?subscribed_fields=messages,messaging_postbacks,message_echoes&access_token=" . urlencode($page_access_token);
    
    $ch = curl_init($webhook_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    curl_close($ch);
    
    // Success! Redirect to inboxes page
    $_SESSION['success_message'] = ucfirst($channel_type) . ' başarıyla bağlandı: ' . $inbox_name;
    header('Location: ../../inboxes.php');
    exit();
    
} catch (Exception $e) {
    error_log("Facebook OAuth Error: " . $e->getMessage());
    die("Error: " . htmlspecialchars($e->getMessage()) . "<br><br><a href='../../inboxes.php'>Back to Inboxes</a>");
}

