<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'includes/workspace.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user = getUserById($user_id);

// Only admin can manage workspaces
if ($user['role'] !== 'admin') {
    header('Location: index.php');
    exit();
}

$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $name = sanitizeInput($_POST['name']);
                $domain = sanitizeInput($_POST['domain']);
                
                if (empty($name)) {
                    $error = 'Workspace adı gerekli';
                } else {
                    $workspace_id = createWorkspace($name, $domain, $user_id);
                    if ($workspace_id) {
                        $message = 'Workspace başarıyla oluşturuldu!';
                    } else {
                        $error = 'Workspace oluşturulamadı';
                    }
                }
                break;
                
            case 'add_user':
                $workspace_id = (int)$_POST['workspace_id'];
                $target_user_id = (int)$_POST['user_id'];
                $role = sanitizeInput($_POST['role']);
                
                // Check if user exists
                $stmt = $pdo->prepare("SELECT id, name, email FROM users WHERE id = ? AND status = 'active'");
                $stmt->execute([$target_user_id]);
                $target_user = $stmt->fetch();
                
                if ($target_user) {
                    if (addUserToWorkspace($target_user['id'], $workspace_id, $role)) {
                        $message = 'Kullanıcı workspace\'e eklendi!';
                    } else {
                        $error = 'Kullanıcı eklenemedi (zaten üye olabilir)';
                    }
                } else {
                    $error = 'Kullanıcı bulunamadı';
                }
                break;
        }
    }
}

// Get all workspaces (admin sees all active workspaces)
$stmt = $pdo->prepare("SELECT * FROM agencies WHERE status = 'active' ORDER BY name ASC");
$stmt->execute();
$all_workspaces = $stmt->fetchAll();

// Get all users for the dropdown
$stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE status = 'active' ORDER BY name ASC");
$stmt->execute();
$all_users = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Workspace Yönetimi - MultiCRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/sidebar.php'; ?>
    
    <div class="main-content">
        <div class="container-fluid" style="padding: 20px;">
            <!-- Page Header -->
            <div class="page-header-mini" style="margin-bottom: 30px;">
                <div style="display: flex; justify-content: space-between; align-items: center; gap: 20px;">
                    <div>
                        <h2 style="margin: 0; display: flex; align-items: center; gap: 12px;">
                            <i class="fas fa-building"></i>
                            <span>Workspace Yönetimi</span>
                        </h2>
                        <p class="text-muted" style="margin: 5px 0 0 0; font-size: 14px;">Workspace'leri oluşturun ve yönetin</p>
                    </div>
                </div>
            </div>
            
            <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <div class="row">
                <!-- Create New Workspace -->
                <div class="col-md-6 mb-4">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0"><i class="fas fa-plus-circle"></i> Yeni Workspace Oluştur</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="create">
                                
                                <div class="mb-3">
                                    <label for="name" class="form-label">Workspace Adı *</label>
                                    <input type="text" class="form-control" id="name" name="name" required>
                                    <small class="text-muted">Örn: Şirket Adı, Departman Adı</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="domain" class="form-label">Domain (Opsiyonel)</label>
                                    <input type="text" class="form-control" id="domain" name="domain">
                                    <small class="text-muted">Örn: sirket.com</small>
                                </div>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Oluştur
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
                
                <!-- Add User to Workspace -->
                <div class="col-md-6 mb-4">
                    <div class="card">
                        <div class="card-header bg-success text-white">
                            <h5 class="mb-0"><i class="fas fa-user-plus"></i> Kullanıcı Ekle</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="add_user">
                                
                                <div class="mb-3">
                                    <label for="workspace_id" class="form-label">Workspace *</label>
                                    <select class="form-select" id="workspace_id" name="workspace_id" required>
                                        <option value="">Seçin...</option>
                                        <?php foreach ($all_workspaces as $ws): ?>
                                            <option value="<?php echo $ws['id']; ?>">
                                                <?php echo htmlspecialchars($ws['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="user_id" class="form-label">Kullanıcı *</label>
                                    <select class="form-select" id="user_id" name="user_id" required>
                                        <option value="">Kullanıcı seçin...</option>
                                        <?php foreach ($all_users as $u): ?>
                                            <option value="<?php echo $u['id']; ?>">
                                                <?php echo htmlspecialchars($u['name']); ?> 
                                                (<?php echo htmlspecialchars($u['email']); ?>) 
                                                - <?php echo ucfirst($u['role']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">Sistemde kayıtlı aktif kullanıcılardan birini seçin</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="role" class="form-label">Rol *</label>
                                    <select class="form-select" id="role" name="role" required>
                                        <option value="agent">Agent</option>
                                        <option value="admin">Admin</option>
                                        <option value="owner">Owner</option>
                                    </select>
                                </div>
                                
                                <button type="submit" class="btn btn-success">
                                    <i class="fas fa-user-plus"></i> Kullanıcı Ekle
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Workspace List -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-list"></i> Tüm Workspace'ler</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Workspace Adı</th>
                                    <th>Domain</th>
                                    <th>Durum</th>
                                    <th>Üye Sayısı</th>
                                    <th>Oluşturulma</th>
                                    <th>İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($all_workspaces as $ws): ?>
                                    <?php
                                    $members = getWorkspaceMembers($ws['id']);
                                    ?>
                                    <tr>
                                        <td><?php echo $ws['id']; ?></td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($ws['name']); ?></strong>
                                        </td>
                                        <td><?php echo htmlspecialchars($ws['domain'] ?? '-'); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $ws['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                                <?php echo ucfirst($ws['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo count($members); ?> üye</td>
                                        <td><?php echo date('d.m.Y', strtotime($ws['created_at'])); ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-info" onclick="showMembers(<?php echo $ws['id']; ?>)" title="Üyeleri Göster">
                                                <i class="fas fa-users"></i>
                                            </button>
                                            <button class="btn btn-sm btn-primary" onclick="editWorkspace(<?php echo $ws['id']; ?>)" title="Düzenle">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-sm btn-danger" onclick="deleteWorkspace(<?php echo $ws['id']; ?>)" title="Sil">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    
                                    <!-- Members Row (Hidden by default) -->
                                    <tr id="members-<?php echo $ws['id']; ?>" style="display: none;">
                                        <td colspan="7">
                                            <div class="p-3 bg-light">
                                                <h6>Workspace Üyeleri:</h6>
                                                <ul class="list-group">
                                                    <?php foreach ($members as $member): ?>
                                                        <li class="list-group-item d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <strong><?php echo htmlspecialchars($member['name']); ?></strong>
                                                                <br>
                                                                <small class="text-muted"><?php echo htmlspecialchars($member['email']); ?></small>
                                                            </div>
                                                            <span class="badge bg-primary"><?php echo ucfirst($member['role']); ?></span>
                                                        </li>
                                                    <?php endforeach; ?>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Workspace Modal -->
    <div class="modal fade" id="editWorkspaceModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Workspace Düzenle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="editWorkspaceForm">
                        <input type="hidden" id="edit_workspace_id">
                        <div class="mb-3">
                            <label for="edit_name" class="form-label">Workspace Adı *</label>
                            <input type="text" class="form-control" id="edit_name" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit_domain" class="form-label">Domain (Opsiyonel)</label>
                            <input type="text" class="form-control" id="edit_domain">
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-primary" onclick="updateWorkspace()">
                        <i class="fas fa-save"></i> Güncelle
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Workspace Modal -->
    <div class="modal fade" id="deleteWorkspaceModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle"></i> Workspace Sil
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i>
                        <strong>Dikkat!</strong> Bu işlem geri alınamaz.
                    </div>
                    <p>Bu workspace'i silmek istediğinizden emin misiniz?</p>
                    <p class="text-muted">
                        <strong id="delete_workspace_name"></strong> workspace'i ve içindeki tüm veriler silinecektir.
                    </p>
                    <input type="hidden" id="delete_workspace_id">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-danger" onclick="confirmDeleteWorkspace()">
                        <i class="fas fa-trash"></i> Evet, Sil
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    function showMembers(workspaceId) {
        const row = document.getElementById('members-' + workspaceId);
        if (row.style.display === 'none') {
            row.style.display = 'table-row';
        } else {
            row.style.display = 'none';
        }
    }
    
    // Edit workspace
    async function editWorkspace(workspaceId) {
        try {
            // Get workspace data from table
            const row = document.querySelector(`[onclick*="showMembers(${workspaceId})"]`).closest('tr');
            const cells = row.cells;
            
            const name = cells[1].querySelector('strong').textContent.trim();
            const domain = cells[2].textContent.trim();
            
            // Fill modal
            document.getElementById('edit_workspace_id').value = workspaceId;
            document.getElementById('edit_name').value = name;
            document.getElementById('edit_domain').value = domain === '-' ? '' : domain;
            
            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('editWorkspaceModal'));
            modal.show();
        } catch (error) {
            console.error('Error:', error);
            alert('Workspace bilgileri alınamadı');
        }
    }
    
    // Update workspace
    async function updateWorkspace() {
        const workspaceId = document.getElementById('edit_workspace_id').value;
        const name = document.getElementById('edit_name').value.trim();
        const domain = document.getElementById('edit_domain').value.trim();
        
        if (!name) {
            alert('Workspace adı gerekli');
            return;
        }
        
        try {
            const response = await fetch('api/workspaces/update.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    workspace_id: parseInt(workspaceId),
                    name: name,
                    domain: domain || null
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                alert(data.message);
                location.reload();
            } else {
                alert('Hata: ' + data.error);
            }
        } catch (error) {
            console.error('Error:', error);
            alert('Bir hata oluştu');
        }
    }
    
    // Delete workspace
    function deleteWorkspace(workspaceId) {
        // Get workspace name from table
        const row = document.querySelector(`[onclick*="showMembers(${workspaceId})"]`).closest('tr');
        const name = row.cells[1].querySelector('strong').textContent.trim();
        
        // Fill modal
        document.getElementById('delete_workspace_id').value = workspaceId;
        document.getElementById('delete_workspace_name').textContent = name;
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('deleteWorkspaceModal'));
        modal.show();
    }
    
    // Confirm delete workspace
    async function confirmDeleteWorkspace() {
        const workspaceId = document.getElementById('delete_workspace_id').value;
        
        if (!workspaceId) {
            alert('Workspace ID bulunamadı');
            return;
        }
        
        // Disable button to prevent double-click
        const deleteBtn = event.target;
        deleteBtn.disabled = true;
        deleteBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Siliniyor...';
        
        try {
            console.log('Silme isteği gönderiliyor:', workspaceId);
            
            const response = await fetch('api/workspaces/delete.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    workspace_id: parseInt(workspaceId)
                })
            });
            
            console.log('Response status:', response.status);
            
            let data;
            const contentType = response.headers.get('content-type');
            
            if (contentType && contentType.includes('application/json')) {
                data = await response.json();
                console.log('Response data:', data);
            } else {
                const text = await response.text();
                console.error('Non-JSON response:', text);
                throw new Error('Sunucudan geçersiz yanıt alındı');
            }
            
            if (data.success) {
                alert(data.message || 'Workspace başarıyla silindi');
                location.reload();
            } else {
                alert('Hata: ' + (data.error || 'Bilinmeyen hata'));
                deleteBtn.disabled = false;
                deleteBtn.innerHTML = '<i class="fas fa-trash"></i> Evet, Sil';
            }
        } catch (error) {
            console.error('Delete error:', error);
            alert('Bir hata oluştu: ' + error.message);
            deleteBtn.disabled = false;
            deleteBtn.innerHTML = '<i class="fas fa-trash"></i> Evet, Sil';
        }
    }
    </script>
</body>
</html>
